<?php
/**
 * @copyright  Copyright (C) 2026 CarAds. All rights reserved.
 * @version 2026.01.20082526
 * @author CarAds Team
 */
namespace CarAdsNextgen\Content;

use function CarAdsNextgen\factory;

class UpdateConfig
{
    static function init($cid): array
    {
        if($cid === -1){
            // if company id is not set or is -1 then we don't need to update configs
            // because we don't have company id to sync configs
            return [
                'status' => 'invalid.company.id'
            ];
        }

        $currentConfigFiles = array_combine(
            factory()->configs->files(),
            array_map(function($file) {
                return factory()->configs->filePath($file);
            }, factory()->configs->files())
        );

        $configPath = factory()->configs->path();

        $configs = self::fetch($cid);

        if(!is_array($configs)){
            return [
                'status' => 'invalid.response'
            ];
        }
        else if(count($configs) === 0){
            return [
                'status' => 'invalid.response'
            ];
        }

        $errors = [];

        foreach($configs as $config){
            $type = $config->type ?? 'undefined';
            $slug = $config->slug ?? 'undefined';
            $content = $config->config_json ?? '{}';

            $basename = $slug . '_' . $type;
            $filename = $basename . '_config.json';


            if($type === 'sitemap'){
                $basename = 'sitemap';
                $filename = 'sitemap_config.json';
            }

            if(!file_exists($configPath . $filename)){
                // if file does not exist then create it
                if(@file_put_contents($configPath . $filename, $content) === false){
                    $errors[$basename] = error_get_last();
                }
            } else {
                // if file exists then update it
                if(@file_put_contents($configPath . $filename, $content) === false){
                    // if file not updated then return error
                    $errors[$basename] = error_get_last();
                }

                if(isset($currentConfigFiles[$basename])){
                    unset($currentConfigFiles[$basename]);
                }
            }
        }

        if(count($errors) > 0){
            return [
                'status' => 'configs.update.failed',
                'errors' => $errors
            ];
        }

        $errors = [];

        // remove all old config files that are not in the new configs
        foreach($currentConfigFiles as $file => $path){
            if(file_exists($path)){
                if(!@unlink($path)){
                    // get last error
                    $errors[$file] = error_get_last();
                }
            }
        }

        if(count($errors) > 0){
            return [
                'status' => 'configs.cleanup.failed',
                'errors' => $errors
            ];
        }

        /**
         *
         */

        if(function_exists('wpo_cache_flush')){
            wpo_cache_flush();
        }

        $files = Cleanup::cache();

        return [
            'status' => 'configs.updated',
            'cache_files_count' => count($files),
        ];
    }

    private static function errorMessage($error): string
    {
        return match($error) {
            'invalid.company.id' => 'Virksomheds-ID er ikke angivet eller er ugyldigt.',
            'invalid.response' => 'Ugyldigt svar fra API\'et.',
            'configs.update.failed' => 'Kunne ikke opdatere konfigurationer. Se fejl nedenfor.',
            'configs.cleanup.failed' => 'Kunne ikke rydde op i gamle konfigurationsfiler. Se fejl nedenfor.',
            'configs.updated' => 'Konfigurationer opdateret.',
            default => 'Der opstod en ukendt fejl.'
        };
    }


    static function adminNotices($response): void
    {
        if(!is_array($response)){
            return;
        }

        $status = $response['status'] ?? 'unknown';
        $errorMessage = self::errorMessage($status);
        $errors = $response['errors'] ?? [];

        if($status === 'configs.updated'){
            echo '<div class="notice notice-success is-dismissible">
                <p>' . $errorMessage . '</p>
                <p>Cache files count: ' . ($response['cache_files_count'] ?? 0) . '</p>
            </div>';
        } else {
            echo '<div class="notice notice-error is-dismissible"><p>' . $errorMessage . '</p>';
            if(count($errors) > 0){
                echo '<ul>';
                foreach($errors as $file => $error){
                    echo '<li>' . $file . ': ' . $error['message'] . '</li>';
                }
                echo '</ul>';
            }
            echo '</div>';
        }

    }

    private static function fetch(string|int $cid){
        $url = "https://nextgen.carads.io/virtuelconfig/list/{$cid}";

        $data = (new \WP_Http())->get($url);
        $response = $data['response'] ?? null;

        if ($response && $response['code'] === 200) {
            $body = json_decode($data['body'] ?? []);


            return $body->configs ?? [];
        }

        return null;
    }
}