<?php
/**
 * @copyright  Copyright (C) 2026 CarAds. All rights reserved.
 * @version 2026.01.20082526
 * @author CarAds Team
 */
namespace CarAdsNextgen\Controller;

use CarAdsNextgen\Utils\Cache;
use stdClass;
use function CarAdsNextgen\Basic\fetch_pre_rendered;
use function CarAdsNextgen\factory;

class ProductController extends BasicProductController
{
    function render() : void
    {
        $route  = $this->route;
        $config = $this->fixSettings(factory()->configs->get($route->active()->path . '_master'));
        $productId  = $route->active()->params->id;
        $typeId     = match($route->active()->params->type ?? '-'){
            '~' => 'carads',
            '-' => 'platform',
            default => 'unknown'
        };
        $clientId   = $config?->cid ?? null;
        $debug      = is_user_logged_in();
        $namespace  = $route->active()->namespace;
        $noCache    = $config->no_cache ?? false;
        $pre_render = $config->pre_render ?? true;


        // check if $productId only has number and not string

        if(!$productId || $config === null)
        {
            $this->seo->robots = 'noindex, nofollow';
            $this->query->set_404();
            status_header(404);
            return;
        }

        // same happend in: product.js:449 (noCache === true && urlType.value === Types.API_ID)
        if ($noCache === true && $typeId === 'platform')
        {
            $noCache = false;
        }

        $product = Cache::byReturn(function() use($config, $productId, $typeId, $clientId, $noCache){
            return $this->fetch($productId, $typeId, $clientId, $noCache);
        }, !$debug ? ($productId . '.json') : null, (60*5), Cache::ENCODE_JSON);

        if ($product->_status === 'error')
        {
            $this->seo->robots = 'noindex, nofollow';
            $this->query->set_404();
            status_header(404);
            return;
        }

        list($before, $after) = $this->page($config, $namespace);

        if($before){
            $before = $this->wrap('carads-intro-top !ca-hidden', str_replace('h1', 'h2', $before), true);
        }

        if($after){
            $after = $this->wrap('carads-intro-bottom !ca-hidden', str_replace('h1', 'h2', $after), true);
        }

        $this->query->is_404 = false;
        $this->title($product, $config);
        $this->description($product, $config);
        $this->canonical($product, $config);
        $this->image($product, $config);
        $this->seo->robots = ($config?->robots_car ?? 'noindex, nofollow');


        status_header(200, 'OK');

        $config->debug = false;

        if(is_user_logged_in()){
            // enable debug mode in javascript
            $config->debug = true;
        }

        $body = '';

        if($pre_render &&  $typeId === 'platform'){
            $body = $this->pre_render($config->cid, $route->active()->path, $productId);
        }

        $this->staticRender($config, $before, $after, $body);
        die;
    }

    private function title(stdClass $product, stdClass $config) : void{
        $title = "{$product->details->brand} {$product->details->model} {$product->details->variant}";
        if (isset($config->title_prefix)){
            $title = "{$config->title_prefix} {$title}";
        }
        else if (isset($config->title_suffix)){
            $title = "{$title} {$config->title_suffix}";
        }

        $this->seo->title = $title;
    }

    private function description(stdClass $product, stdClass $config) : void{
        $description = $product?->description ?? 'biler til salg';

        if(isset($config->description_prefix)){
            $description = "{$config->description_prefix} {$description}";
        }
        else if(isset($config->description_suffix)){
            $description = "{$description} {$config->description_suffix}";
        }

        $this->seo->description = $description;
    }

    private function canonical(stdClass $product, stdClass $config) : void{
        $urls = $product?->url ?? null;

        if($urls === null)
        {
            return;
        }

        $a = current($urls);
        $b = $a?->prices ? current($a->prices) : null;

        $url = $b?->url ?? null;

        if($url && !str_contains($url, 'debug')){
            $this->seo->canonical = rtrim($url, '/');
        }
    }

    private function image(stdClass $product, stdClass $config): void
    {
        $this->seo->images = array_map(function($image){
            return "https://nextgen.carads.io/media/{$image->id}";
        }, $product?->media?->images ?? []);
    }

    private function fetch(
        int $id,
        string $type = '-',
        ?int $company_id = null,
        ?bool $noCache = false
    ){
        $base = "https://nextgen.carads.io/products/{$company_id}";

        if($noCache){
            $base = "https://nextgen.carads.io/product/nocache/{$company_id}";
        }

        try{
            $url = "{$base}/{$id}";

            if($type !== 'carads')
            {
                $url = "{$base}/x{$id}";
            }

            $data = (new \WP_Http())->get($url);

            $response   = (object) ($data['response'] ?? ['code' => 500]);
            $body       = json_decode($data['body'] ?? []);
            $body->product = $body->product->item ?? new \stdClass;
        }
        catch (\Exception $e){
            $response   = (object) ['code' => 500];
            $body       = new \stdClass;
        }

        if ($response->code === 200 && ($body?->created ?? null) !== null)
        {
            $body->_status = 'success';
        }
        else
        {
            $body = (object) [
                '_status' => 'error',
                '_message' => $body->message ?? 'Unknown error',
            ];
        }

        return $body;
    }

    private function pre_render(int $cid, string $name, int $id) : string{
        return fetch_pre_rendered('https://ssr.js.nextgen.carads.io/product/' . $cid . '/' . $name . '/' . $id);
    }
}