<?php
/**
 * @copyright  Copyright (C) 2026 CarAds. All rights reserved.
 * @version 2026.01.20082526
 * @author CarAds Team
 */
namespace CarAdsNextgen\Controller;

use CarAdsNextgen\Utils\Cache;
use function CarAdsNextgen\Basic\fetch_pre_rendered;
use function CarAdsNextgen\factory;

class ProductsController extends BasicProductController
{
    public function render() : void
    {
        $route      = $this->route;
        $params     = $route->active()->params;
        $namespace  = $this->route->active()->path;
        $query      = $_REQUEST;

        $config = $this->fixSettings(factory()->configs->get($route->active()->path . '_master'));

        $title = $config?->title ?? 'Biler til salg';
        $description = $config?->description ?? 'Biler til salg';
        $robots = $config->robots ?? 'follow,index';
        $data = null;
        $pre_render = true;

        // support for debug mode with ?debug=1 - so we can see what is wrong
        // it ask for pass and username like auth basic
        $debug = is_user_logged_in();

        if ('product-list-filter' === $route->active()->controller)
        {
            $brand = $params?->brand ?? null;
            $model = $params?->model ?? null;

            if($brand){
                $brand = urldecode($brand);
                $query['brand'] = $brand;
            }

            if($model){
                $model = urldecode($model);
                $query['model'] = $model;
            }

            $brandKey = $brand ?? 'null';
            $modelKey = $model ?? 'null';

            $filename = "{$brandKey}-{$modelKey}-{$config->cid}-{$config->fid}";

            if($debug){
                $filename = null;
            }

            $match = Cache::byReturn(function() use($config, $brand, $model){
                return $this->fetch($config->cid, $config->fid, $brand, $model);
            }, $filename, 3600, Cache::ENCODE_JSON);

            $correctItem = (array) ($match?->content?->{$namespace} ?? $match?->conten?->default ?? []);

            if(count($correctItem) > 0)
            {
                $this->query->is_404 = false;
                $data = (object) $correctItem;
                $data->brand = $brand ?? null;
                $data->model = $model ?? null;
                $data->key = "{$brand}-{$model}";

                $title       = $config->current_title = $data?->website_title ?? $data?->title ?? $title;
                $description = $config->current_description = $data?->website_description ?? $data?->description ?? $description;
            }
            else if($brand && $model)
            {
                $models = (array) ($match?->aggregates?->model ?? []);
                $brands = (array) ($match?->aggregates?->brand ?? []);

                if(!($match?->aggregates?->model) && !($match?->aggregates?->brand))
                {
                    // both brand and model is not correct - so it 404
                    $this->query->is_404 = true;
                }
                // not sure this fix around ?? '' is correct but waning is gone.
                else if(count($models) !== 0 && !($match?->aggregates?->brand ?? ''))
                {
                    // brand is correct but model not - so it 404
                    $this->query->is_404 = true;
                }
                else if(count($models) !== 0 && count($brands) !== 0)
                {
                    // both brand and model is correct - so it not 404
                    $this->query->is_404 = false;
                }
                else
                {
                    return;
                }
            }
            else{
                // only brand
                if(!($match?->aggregates?->brand ?? null))
                {
                    $this->query->is_404 = true;
                }
                else
                {
                    $this->query->is_404 = false;
                }
            }
        }
        else
        {
            $this->query->is_404 = false;
        }

        $this->seo->title = $title;
        $this->seo->description = $description;
        $this->seo->robots = $robots;

        if(!$this->query->is_404){
            $this->seo->canonical = strtolower(rtrim((@explode('?', @home_url($_SERVER['REQUEST_URI']))[0]), '/'));
        }

        $before = null;
        $after = null;

        if('product-list-filter' !== $route->active()->controller){
            list($before, $after) = $this->page($config, $namespace);

            if($before){
                $before = $this->wrap('carads-intro-top', $before);
            }

            if($after){
                $after = $this->wrap('carads-intro-bottom', $after);
            }
        }
        elseif($data !== null){
            $config->tag = 'a';
            $before = $this->callSearch('before', $config, $data, $namespace);
            $after = $this->callSearch('after', $config, $data, $namespace);
        }

        $config->debug = false;

        if(is_user_logged_in()){
            // enable debug mode in javascript
            $config->debug = true;
        }

        $config->robots = $config->robots ?? 'follow,index';

        status_header(($this->query->is_404 ? 404 : 200), ($this->query->is_404 ? 'Not Found' : 'OK'));



        $body = '';
        $page_number = 1;
        $filter_brand = $query['brand'] ?? null;
        $filter_model = $query['model'] ?? null;

        if(count($query) > 0){
            $page_number = $query['page'] = (int) ($query['page'] ?? 1);
            $allowed = ['brand', 'model', 'page'];
            foreach($query as $key => $value){
                if(!in_array($key, $allowed)){
                    $pre_render = false;
                    break;
                }
            }
        }

        if($pre_render){
            $body = $this->pre_render(
                cid: $config->cid,
                name: $route->active()->path,
                page_number: $page_number,
                filter_brand: $filter_brand,
                filter_model: $filter_model
            );
        }


        $this->staticRender(
            $config,
            $before,
            $after,
            $body
        );
        die;
    }

    private function fetch(int $cid, int $fid, ?string $brand = null, ?string $model = null): object
    {
        try
        {
            // test url was https://nextgen.carads.io/feeds/25/75/1 - change to dynamic
            $url = "https://nextgen.carads.io/feeds/{$cid}/{$fid}/1";

            $filter = [];

            if($brand)
            {
                $filter['details.brand'] = [
                    '=' => [$brand]
                ];
            }

            if($model)
            {
                $filter['details.model'] = [
                    '=' => [$model]
                ];
            }

            $data = [
                'page' => 1,
                'sort' => 'name:asc',
                'filter' => $filter,
                'order' => [
                    'details.brand' => 'asc',
                    'details.model' => 'asc'
                ],
                'size' => 1
            ];


            $data = (new \WP_Http())->post($url, ['body' => json_encode($data), 'data_format' => 'body']);

            if (!is_array($data) && ($data instanceof \WP_Error)){
                throw new \Exception($data->get_error_message());
            }

            $body       = json_decode($data['body'] ?? [], true);
        }
        catch (\Exception $e)
        {
            $body       = [];
        }

        $aggregates = array_filter($body['aggregates'] ?? [], function($item, $key) {
            return $key === 'details.brand' || $key === 'details.model';
        }, ARRAY_FILTER_USE_BOTH);

        if(isset($aggregates['details.brand']))
        {
            $aggregates['brand'] = $aggregates['details.brand'];
            unset($aggregates['details.brand']);
        }

        if(isset($aggregates['details.model']))
        {
            $aggregates['model'] = $aggregates['details.model'];
            unset($aggregates['details.model']);
        }


        $content = $body['content'] ?? [];

        if($content) {
            $content = array_map(function ($item) {
                $item['list'] = $item['list'] ?? 'default';

                if(empty($item['list'])){
                    $item['list'] = 'default';
                }

                return $item;
            }, array_filter(($content['content'] ?? []), function ($item) {
                return !is_scalar($item);
            }));

            $content = array_combine(array_column($content, 'list'), $content);
        }

        return (object)[
            'aggregates' => (object)$aggregates,
            'content' => $content
        ];
    }

    private function pre_render(int $cid, string $name, $page_number, ?string $filter_brand, ?string $filter_model) : string{

        $query = http_build_query([
            'details.brand' => [$filter_brand],
            'details.model' => [$filter_model],
            'page' => $page_number
        ]);
        $query = preg_replace('/%5B[0-9]+%5D/simU', '%5B%5D', $query);


        return @fetch_pre_rendered('https://ssr.js.nextgen.carads.io/product/' . $cid . '/' . $name . '/?' . $query);
    }
}