<?php
/**
 * @copyright  Copyright (C) 2026 CarAds. All rights reserved.
 * @version 2026.01.20082526
 * @author CarAds Team
 */
namespace CarAdsNextgen\Shortcode;
use CarAdsNextgen\Basic\ElementorWidget;
use CarAdsNextgen\Basic\Shortcode;
use CarAdsNextgen\Content\Enqueue;
use function CarAdsNextgen\Basic\createAttributes;
use function CarAdsNextgen\factory;

/**
 * Class QuickSearch
 */
class SlowSearch extends Shortcode
{
	/**
	 * @var string
	 */
	protected string $tag = 'car-ads-search-slow';

	public function render($args = [], $content = null) : string
	{
        $id = $args['id'] ?? null;
        $sort = $args['sort'] ?? null;
        $clickable = $args['clickable'] ?? null;
        $json = is_string($content) ? trim($content) : null;

        if (!$id)
        {
            return 'You need add id to shortcode';
        }

        if (!factory()->configs->exists($id . '_slow'))
        {
            return 'no config found...';
        }

        $config = factory()->configs->getRaw($id . '_slow');

        $config = json_decode($config);

        if($sort){
            $config->sort = $sort;
        }

        if($clickable){
            if (!isset($config->search)){
               $config->search = new \stdClass();
            }

            $config->search->clickable = $clickable;
        }

        if($json){
            $json = json_decode($json);

            /**
             * What is query?
             * is same as filter in all other apps - for filter in this app use for user input (top filter)
             */
            if($json->query){
                $config->query = $json->query;
            }

            if($json->filter){
                $config->filter = $json->filter;
            }
        }

        if(factory()->settings->cid !== -1){
            $config->cid = factory()->settings->cid; // overwrite the client id for the config
        }

        // update need in js
        $uri            = $_SERVER['REDIRECT_URL'] ?? $_SERVER['REQUEST_URI'];
        $path           = parse_url($uri, PHP_URL_PATH);
        $config->base   = rtrim($path, '/');

        if (!isset($config->pageKey))
        {
            $config->pageKey = '_page';
        }

        $config = json_encode($config);

        return $this->inlineHTML($config, $id);
	}

    public function enqueue():void{
        if(factory()->settings->optimize){
            Enqueue::css(Enqueue::SLOW_SEARCH);
        }
    }

    private function inlineHTML(string $config, string|int $id):string{
        $time = CA_TIME;
        return <<<HTML
            <div class="carads-wrapper">
                <div class="carads" data-time="{$time}">
                    <div id="app-slow" data-version="2" class="ca-relative ca-min-h-screen">
                    </div>
                </div>
            </div>
            <script type="application/json" id="carads_search_settings">
                {$config}
            </script>
HTML;
    }

    public function elementor_register() : void{

        add_action('elementor/widgets/register', function($widgets_manager) {
            $widgets_manager->register(new class extends ElementorWidget{
                protected string $title = 'CarAds Slow Search';
                protected string $tag = 'CarAdsSlowSearch';
                protected string $icon = 'eicon-search';
                protected array $category = ['carads-widgets'];

                protected function register_controls(): void
                {
                    $this->start_controls_section(
                        'section_content',
                        [
                            'label' => __('Content', 'CarAdsNextgen'),
                        ]
                    );
                    $this->add_control(
                        'id',
                        [
                            'label' => __('Search ID', 'CarAdsNextgen'),
                            'type' => \Elementor\Controls_Manager::TEXT,
                            'default' => '',
                        ]
                    );
                    $this->add_control(
                        'sort',
                        [
                            'label' => __('Sort', 'CarAdsNextgen'),
                            'type' => \Elementor\Controls_Manager::TEXT,
                            'default' => '',
                        ]
                    );
                    $this->add_control(
                        'clickable',
                        [
                            'label' => __('Clickable', 'CarAdsNextgen'),
                            'type' => \Elementor\Controls_Manager::SWITCHER,
                            'label_on' => __('Yes', 'CarAdsNextgen'),
                            'label_off' => __('No', 'CarAdsNextgen'),
                            'return_value' => 'yes',
                            'default' => 'yes',
                        ]
                    );
                    $this->add_control('pre_render',[
                        'label' => __('Use Pre-rendered Content', 'CarAdsNextgen'),
                        'type' => \Elementor\Controls_Manager::SWITCHER,
                        'label_on' => __('Yes', 'CarAdsNextgen'),
                        'label_off' => __('No', 'CarAdsNextgen'),
                        'return_value' => 'yes',
                        'default' => 'yes',
                        'description' => __('Enable to use pre-rendered content for better performance.', 'CarAdsNextgen')
                    ]);

                    $this->add_control(
                        'json',
                        [
                            'label' => __('JSON Filter/Query', 'CarAdsNextgen'),
                            'type' => \Elementor\Controls_Manager::TEXTAREA,
                            'default' => '',
                            'description' => __('Enter JSON formatted filter and/or query to predefine search results.', 'CarAdsNextgen')
                        ]
                    );
                    $this->end_controls_section();
                }

                protected function render() : void{
                    $s = $this->get_settings_for_display();
                    $attr = [
                        'id' => $s['id'] ?? '',
                        'sort' => $s['sort'] ?? '',
                        'clickable' => ($s['clickable'] ?? '') === 'yes' ? 1 : 0,
                        'pre_render' => ($s['pre_render'] ?? '') === 'yes' ? 1 : 0,
                    ];

                    $attr = array_filter($attr, function($value){
                        return !empty($value);
                    });

                    $attr = createAttributes($attr);

                    $json = empty($s['json']) ? null : trim($s['json']);

                    echo do_shortcode('[car-ads-search-slow ' . $attr . ']' . $json . '[/car-ads-search-slow]');
                }
            });
        });
    }
}