<?php
/**
 * @copyright  Copyright (C) 2026 CarAds. All rights reserved.
 * @version 2026.01.20082526
 * @author CarAds Team
 */
namespace CarAdsNextgen\Takeover;
/**
 * Class Hook
 * @package CarAdsNextgen\Takeover
 *
 *
 * - it create for take over the WP_Hook class in case,
 *   we need better control over the hooks in the future.
 */
#[AllowDynamicProperties]
final class Hook implements \Iterator, \ArrayAccess
{
    public array $callbacks = [];

    protected array $priorities = [];

    private array $iterations = [];

    private array $current_priority = [];

    private int $nesting_level = 0;

    private bool $doing_action = false;

    private array $before = [];

    private array $after = [];

    protected bool $output = false;

    protected bool $debugEnabled = false;
    protected array $debug = [];

    static public function take(string $hook_name) : Hook{
        global $wp_filter;

        if(!isset($wp_filter[$hook_name])){
            $hook = new self();
            $wp_filter[$hook_name] = $hook;
            return $hook;
        }
        else if($wp_filter[$hook_name] instanceof self){
            return $wp_filter[$hook_name];
        }
        else if($wp_filter[$hook_name] instanceof \WP_Hook){
            $hook = self::override($wp_filter[$hook_name]);
            $wp_filter[$hook_name] = $hook;
            return $hook;
        }
    }

    static public function override(\WP_Hook $hook) : self
    {
        $cl = new \ReflectionClass($hook);
        $_properties = $cl->getProperties();
        $properties = array_map(function($property) use ($hook) {
            $property->setAccessible(true);
            return [
                'name' => $property->getName(),
                'value' => $property->getValue($hook)
            ];
        }, $_properties);

        $properties = array_combine(
            array_column($properties, 'name'),
            array_column($properties, 'value')
        );

        $hook = new self();
        $hook->callbacks            = $properties['callbacks'];
        $hook->priorities           = $properties['priorities'];
        $hook->iterations           = $properties['iterations'];
        $hook->current_priority     = $properties['current_priority'];
        $hook->nesting_level        = $properties['nesting_level'];
        $hook->doing_action         = $properties['doing_action'];

        return $hook;
    }

    public function enable_debug(bool $bool = true) : void{
        $this->debugEnabled = $bool;
    }

    public function getDebug() : array{
        return $this->debug;
    }

    public function add_before(callable $callback) : void{
        $this->before[] = $callback;
    }

    public function add_after(callable $callback) : void{
        $this->after[] = $callback;
    }

    private function do_before() : void{
        foreach($this->before as $before){
            call_user_func($before);
        }
    }

    public function setOutput(bool $output) : void{
        $this->output = $output;
    }

    private function do_after(&$value, &$output) : void{
        foreach($this->after as $after){
            if($this->output){
                call_user_func_array($after, [&$value, &$output]);
            }
            else{
                call_user_func_array($after, [&$value]);
            }
        }
    }

    /**
     * Adds a callback function to a filter hook.
     *
     * @since 4.7.0
     *
     * @param string   $hook_name     The name of the filter to add the callback to.
     * @param callable|string $callback      The callback to be run when the filter is applied.
     * @param int      $priority      The order in which the functions associated with a particular filter
     *                                are executed. Lower numbers correspond with earlier execution,
     *                                and functions with the same priority are executed in the order
     *                                in which they were added to the filter.
     * @param int      $accepted_args The number of arguments the function accepts.
     */
    public function add_filter(string $hook_name, callable|string $callback, int $priority, int $accepted_args ) : void {
        $idx = _wp_filter_build_unique_id( $hook_name, $callback, $priority );

        $priority_existed = isset( $this->callbacks[ $priority ] );

        $this->callbacks[ $priority ][ $idx ] = array(
            'function'      => $callback,
            'accepted_args' => (int) $accepted_args,
        );

        // If we're adding a new priority to the list, put them back in sorted order.
        if ( ! $priority_existed && count( $this->callbacks ) > 1 ) {
            ksort( $this->callbacks, SORT_NUMERIC );
        }

        $this->priorities = array_keys( $this->callbacks );

        if ( $this->nesting_level > 0 ) {
            $this->resort_active_iterations( $priority, $priority_existed );
        }
    }

    /**
     * Handles resetting callback priority keys mid-iteration.
     *
     * @since 4.7.0
     *
     * @param false|int $new_priority     Optional. The priority of the new filter being added. Default false,
     *                                    for no priority being added.
     * @param bool      $priority_existed Optional. Flag for whether the priority already existed before the new
     *                                    filter was added. Default false.
     */
    private function resort_active_iterations(false|int $new_priority = false, bool $priority_existed = false ) : void {
        $new_priorities = $this->priorities;

        // If there are no remaining hooks, clear out all running iterations.
        if ( ! $new_priorities ) {
            foreach ( $this->iterations as $index => $iteration ) {
                $this->iterations[ $index ] = $new_priorities;
            }

            return;
        }

        $min = min( $new_priorities );

        foreach ( $this->iterations as $index => &$iteration ) {
            $current = current( $iteration );

            // If we're already at the end of this iteration, just leave the array pointer where it is.
            if ( false === $current ) {
                continue;
            }

            $iteration = $new_priorities;

            if ( $current < $min ) {
                array_unshift( $iteration, $current );
                continue;
            }

            while ( current( $iteration ) < $current ) {
                if ( false === next( $iteration ) ) {
                    break;
                }
            }

            // If we have a new priority that didn't exist, but ::apply_filters() or ::do_action() thinks it's the current priority...
            if ( $new_priority === $this->current_priority[ $index ] && ! $priority_existed ) {
                /*
                 * ...and the new priority is the same as what $this->iterations thinks is the previous
                 * priority, we need to move back to it.
                 */

                if ( false === current( $iteration ) ) {
                    // If we've already moved off the end of the array, go back to the last element.
                    $prev = end( $iteration );
                } else {
                    // Otherwise, just go back to the previous element.
                    $prev = prev( $iteration );
                }

                if ( false === $prev ) {
                    // Start of the array. Reset, and go about our day.
                    reset( $iteration );
                } elseif ( $new_priority !== $prev ) {
                    // Previous wasn't the same. Move forward again.
                    next( $iteration );
                }
            }
        }

        unset( $iteration );
    }

    /**
     * Removes a callback function from a filter hook.
     *
     * @since 4.7.0
     *
     * @param string                $hook_name The filter hook to which the function to be removed is hooked.
     * @param callable|string|array $callback  The callback to be removed from running when the filter is applied.
     *                                         This method can be called unconditionally to speculatively remove
     *                                         a callback that may or may not exist.
     * @param int                   $priority  The exact priority used when adding the original filter callback.
     * @return bool Whether the callback existed before it was removed.
     */
    public function remove_filter(string $hook_name, string|callable|array $callback, int $priority ) : bool {
        $function_key = _wp_filter_build_unique_id( $hook_name, $callback, $priority );

        $exists = isset( $this->callbacks[ $priority ][ $function_key ] );

        if ( $exists ) {
            unset( $this->callbacks[ $priority ][ $function_key ] );

            if ( ! $this->callbacks[ $priority ] ) {
                unset( $this->callbacks[ $priority ] );

                $this->priorities = array_keys( $this->callbacks );

                if ( $this->nesting_level > 0 ) {
                    $this->resort_active_iterations();
                }
            }
        }

        return $exists;
    }

    /**
     * Checks if a specific callback has been registered for this hook.
     *
     * When using the `$callback` argument, this function may return a non-boolean value
     * that evaluates to false (e.g. 0), so use the `===` operator for testing the return value.
     *
     * @since 4.7.0
     *
     * @param string                      $hook_name Optional. The name of the filter hook. Default empty.
     * @param callable|string|array|false $callback  Optional. The callback to check for.
     *                                               This method can be called unconditionally to speculatively check
     *                                               a callback that may or may not exist. Default false.
     * @return bool|int If `$callback` is omitted, returns boolean for whether the hook has
     *                  anything registered. When checking a specific function, the priority
     *                  of that hook is returned, or false if the function is not attached.
     */
    public function has_filter(string $hook_name = '', callable|string|array|false $callback = false ) : bool|int {
        if ( false === $callback ) {
            return $this->has_filters();
        }

        $function_key = _wp_filter_build_unique_id( $hook_name, $callback, false );

        if ( ! $function_key ) {
            return false;
        }

        foreach ( $this->callbacks as $priority => $callbacks ) {
            if ( isset( $callbacks[ $function_key ] ) ) {
                return $priority;
            }
        }

        return false;
    }

    /**
     * Checks if any callbacks have been registered for this hook.
     *
     * @since 4.7.0
     *
     * @return bool True if callbacks have been registered for the current hook, otherwise false.
     */
    public function has_filters() : bool {
        foreach ( $this->callbacks as $callbacks ) {
            if ( $callbacks ) {
                return true;
            }
        }

        return false;
    }

    /**
     * Removes all callbacks from the current filter.
     *
     * @since 4.7.0
     *
     * @param int|false $priority Optional. The priority number to remove. Default false.
     */
    public function remove_all_filters(int|false $priority = false) : void {
        if ( ! $this->callbacks ) {
            return;
        }

        if ( false === $priority ) {
            $this->callbacks  = array();
            $this->priorities = array();
        } elseif ( isset( $this->callbacks[ $priority ] ) ) {
            unset( $this->callbacks[ $priority ] );
            $this->priorities = array_keys( $this->callbacks );
        }

        if ( $this->nesting_level > 0 ) {
            $this->resort_active_iterations();
        }
    }

    /**
     * Calls the callback functions that have been added to a filter hook.
     *
     * @since 4.7.0
     *
     * @param mixed $value The value to filter.
     * @param array $args  Additional parameters to pass to the callback functions.
     *                     This array is expected to include $value at index 0.
     * @return mixed The filtered value after all hooked functions are applied to it.
     */
    public function apply_filters( $value, $args ) {
        if ( ! $this->callbacks ) {
            return $value;
        }

        $nesting_level = $this->nesting_level++;

        $this->iterations[ $nesting_level ] = $this->priorities;

        $num_args = count( $args );

        $this->do_before();
        $this->output && ob_start();

        do {
            $this->current_priority[ $nesting_level ] = current( $this->iterations[ $nesting_level ] );

            $priority = $this->current_priority[ $nesting_level ];

            foreach ( $this->callbacks[ $priority ] as $the_ ) {
                if ( ! $this->doing_action ) {
                    $args[0] = $value;
                }

                $debugIndex = $this->doDebug($the_['function'], $priority, $args, null);

                // Avoid the array_slice() if possible.
                if ( 0 === $the_['accepted_args'] ) {
                    $value = call_user_func( $the_['function'] );
                } elseif ( $the_['accepted_args'] >= $num_args ) {
                    $value = call_user_func_array( $the_['function'], $args );
                } else {
                    $value = call_user_func_array( $the_['function'], array_slice( $args, 0, $the_['accepted_args'] ) );
                }

                $debugIndex != -1 && $this->doDebug($the_['function'], $priority, $args, $debugIndex);
            }
        } while ( false !== next( $this->iterations[ $nesting_level ] ) );

        unset( $this->iterations[ $nesting_level ] );
        unset( $this->current_priority[ $nesting_level ] );

        --$this->nesting_level;


        $output = null;
        $this->output && $output = ob_get_clean();
        $this->do_after($value, $output);

        if($this->output){
            echo $output;
        }

        return $value;
    }

    /**
     * Calls the callback functions that have been added to an action hook.
     *
     * @since 4.7.0
     *
     * @param array $args Parameters to pass to the callback functions.
     */
    public function do_action(array $args) : void {
        $this->doing_action = true;
        $this->apply_filters( '', $args );

        // If there are recursive calls to the current action, we haven't finished it until we get to the last one.
        if ( ! $this->nesting_level ) {
            $this->doing_action = false;
        }
    }

    /**
     * Processes the functions hooked into the 'all' hook.
     *
     * @since 4.7.0
     *
     * @param array $args Arguments to pass to the hook callbacks. Passed by reference.
     */
    public function do_all_hook(array &$args) : void {
        $nesting_level                      = $this->nesting_level++;
        $this->iterations[ $nesting_level ] = $this->priorities;

        do {
            $priority = current( $this->iterations[ $nesting_level ] );

            foreach ( $this->callbacks[ $priority ] as $the_ ) {
                call_user_func_array( $the_['function'], $args );
            }
        } while ( false !== next( $this->iterations[ $nesting_level ] ) );

        unset( $this->iterations[ $nesting_level ] );
        --$this->nesting_level;
    }

    /**
     * Return the current priority level of the currently running iteration of the hook.
     *
     * @since 4.7.0
     *
     * @return int|false If the hook is running, return the current priority level.
     *                   If it isn't running, return false.
     */
    public function current_priority(): false|int
    {
        if ( false === current( $this->iterations ) ) {
            return false;
        }

        return current( current( $this->iterations ) );
    }

    /**
     * Normalizes filters set up before WordPress has initialized to WP_Hook objects.
     *
     * The `$filters` parameter should be an array keyed by hook name, with values
     * containing either:
     *
     *  - A `WP_Hook` instance
     *  - An array of callbacks keyed by their priorities
     *
     * Examples:
     *
     *     $filters = array(
     *         'wp_fatal_error_handler_enabled' => array(
     *             10 => array(
     *                 array(
     *                     'accepted_args' => 0,
     *                     'function'      => function() {
     *                         return false;
     *                     },
     *                 ),
     *             ),
     *         ),
     *     );
     *
     * @since 4.7.0
     *
     * @param array $filters Filters to normalize. See documentation above for details.
     * @return WP_Hook[] Array of normalized filters.
     */
    public static function build_preinitialized_hooks( $filters ) {
        /** @var WP_Hook[] $normalized */
        $normalized = array();

        foreach ( $filters as $hook_name => $callback_groups ) {
            if ( $callback_groups instanceof WP_Hook ) {
                $normalized[ $hook_name ] = $callback_groups;
                continue;
            }

            $hook = new WP_Hook();

            // Loop through callback groups.
            foreach ( $callback_groups as $priority => $callbacks ) {

                // Loop through callbacks.
                foreach ( $callbacks as $cb ) {
                    $hook->add_filter( $hook_name, $cb['function'], $priority, $cb['accepted_args'] );
                }
            }

            $normalized[ $hook_name ] = $hook;
        }

        return $normalized;
    }

    /**
     * Determines whether an offset value exists.
     *
     * @since 4.7.0
     *
     * @link https://www.php.net/manual/en/arrayaccess.offsetexists.php
     *
     * @param mixed $offset An offset to check for.
     * @return bool True if the offset exists, false otherwise.
     */
    #[ReturnTypeWillChange]
    public function offsetExists(mixed $offset ) : bool {
        return isset( $this->callbacks[ $offset ] );
    }

    /**
     * Retrieves a value at a specified offset.
     *
     * @since 4.7.0
     *
     * @link https://www.php.net/manual/en/arrayaccess.offsetget.php
     *
     * @param mixed $offset The offset to retrieve.
     * @return mixed If set, the value at the specified offset, null otherwise.
     */
    #[ReturnTypeWillChange]
    public function offsetGet(mixed $offset) : mixed {
        return $this->callbacks[$offset] ?? null;
    }

    /**
     * Sets a value at a specified offset.
     *
     * @since 4.7.0
     *
     * @link https://www.php.net/manual/en/arrayaccess.offsetset.php
     *
     * @param mixed $offset The offset to assign the value to.
     * @param mixed $value The value to set.
     */
    #[ReturnTypeWillChange]
    public function offsetSet(mixed $offset, mixed $value) : void {
        if ( is_null( $offset ) ) {
            $this->callbacks[] = $value;
        } else {
            $this->callbacks[ $offset ] = $value;
        }

        $this->priorities = array_keys( $this->callbacks );
    }

    /**
     * Unsets a specified offset.
     *
     * @since 4.7.0
     *
     * @link https://www.php.net/manual/en/arrayaccess.offsetunset.php
     *
     * @param mixed $offset The offset to unset.
     */
    #[ReturnTypeWillChange]
    public function offsetUnset(mixed $offset) : void {
        unset( $this->callbacks[ $offset ] );
        $this->priorities = array_keys( $this->callbacks );
    }

    /**
     * Returns the current element.
     *
     * @since 4.7.0
     *
     * @link https://www.php.net/manual/en/iterator.current.php
     *
     * @return array Of callbacks at current priority.
     */
    #[ReturnTypeWillChange]
    public function current() : array {
        return current( $this->callbacks );
    }

    /**
     * Moves forward to the next element.
     *
     * @since 4.7.0
     *
     * @link https://www.php.net/manual/en/iterator.next.php
     *
     * @return void
     */
    #[ReturnTypeWillChange]
    public function next() : void{
        next( $this->callbacks );
    }

    /**
     * Returns the key of the current element.
     *
     * @return string|int|null Returns current priority on success, or NULL on failure
     * @link https://www.php.net/manual/en/iterator.key.php
     *
     * @since 4.7.0
     */
    #[ReturnTypeWillChange]
    public function key() : string|int|null
    {
        return key( $this->callbacks );
    }

    /**
     * Checks if current position is valid.
     *
     * @since 4.7.0
     *
     * @link https://www.php.net/manual/en/iterator.valid.php
     *
     * @return bool Whether the current position is valid.
     */
    #[ReturnTypeWillChange]
    public function valid() : bool {
        return key( $this->callbacks ) !== null;
    }

    /**
     * Rewinds the Iterator to the first element.
     *
     * @since 4.7.0
     *
     * @link https://www.php.net/manual/en/iterator.rewind.php
     */
    #[ReturnTypeWillChange]
    public function rewind() : void {
        reset( $this->callbacks );
    }

    /**
     * @param mixed $function1
     * @param mixed $priority
     * @param array $args
     * @param int|null $index
     * @return int
     */
    public function doDebug($function1, mixed $priority, array $args, ?int $index = null): int
    {
        if(!$this->debugEnabled || ($index !== null && !isset($this->debug[$index]))){
            return -1;
        }

        if($index !== null && array_key_exists('end', $this->debug[$index])){
            $this->debug[$index]['end'] = floatval(microtime(true));
            if($this->debug[$index]['end'] - $this->debug[$index]['start'] < 0.0001){
                $this->debug[$index]['time'] = floatval(0.0001);
            }
            else{
                $this->debug[$index]['time'] = floatval($this->debug[$index]['end'] - $this->debug[$index]['start']);
            }

            return -1;
        }

        if (is_array($function1)) {
            if(is_object($function1[0])){
                $function = get_class($function1[0]) . '::' . $function1[1];
            }
            else{
                $function = $function1[0] . '::' . $function1[1];
            }
        } else if (is_string($function1)) {
            $function = $function1;
        } else {
            $function = 'anonymous';
        }

        $this->debug[] = [
            'function' => $function,
            'priority' => $priority,
            'args' => $args,
            'start' => floatval(microtime(true)),
            'end' => null,
            'time' => null
        ];

        return count($this->debug) - 1;
    }
}