<?php
/**
 * @copyright  Copyright (C) 2026 CarAds. All rights reserved.
 * @version 2026.01.20082526
 * @author CarAds Team
 */
namespace CarAdsNextgen\Utils;

class Cache
{
    const int ENCODE_JSON = 1;
    const int ENCODE_STRING = 2;
    const int ENCODE_SERIALIZE = 3;
    /**
     * output is used when we need to echo something from function - it used for all render methods from WordPress and
     * other stuff like that.
     *
     * @param callable $callback
     * @param string|null $filename
     * @param int $cacheTime
     * @return string
     */
    public static function output(callable $callback, ?string $filename, int $cacheTime = 3600): string
    {
        /**
         * as long as we are logged in, we don't need to cache for be able to see changes in real time
         */
        if($filename === null || is_user_logged_in())
            return self::callback($callback);

        $cacheFile = CA_CACHE_PATH . '/' . $filename;

        if (file_exists($cacheFile) && filemtime($cacheFile) > time() - $cacheTime)
        {
            return file_get_contents($cacheFile);
        }

        $content = self::callback($callback);

        @file_put_contents($cacheFile, $content);

        return $content;
    }

    /**
     * byReturn is used when we need to return string from function - I think we can use it for cache http requests
     * to api and other stuff.
     **/
    public static function byReturn(
        callable $callback,
        ?string $filename,
        int $cacheTime = 3600,
        int $encode = self::ENCODE_STRING
    ): string|int|array|object
    {
        /**
         * as long as we are logged in, we don't need to cache for be able to see changes in real time
         */
        if($filename === null || is_user_logged_in()) {
            /**
             * we load content from callback so we not need to decode/encode content
             */
            return $callback();
        }

        $cacheFile = CA_CACHE_PATH . '/' . $filename;

        if (file_exists($cacheFile) && filemtime($cacheFile) > time() - $cacheTime)
        {
            $content = file_get_contents($cacheFile);

            return match ($encode) {
                self::ENCODE_JSON => json_decode($content),
                self::ENCODE_SERIALIZE => unserialize($content),
                default => $content
            };
        }

        $content = $callback();

        @file_put_contents($cacheFile, match ($encode) {
            self::ENCODE_JSON => json_encode($content),
            self::ENCODE_SERIALIZE => serialize($content),
            default => $content
        });

        return match ($encode) {
            self::ENCODE_JSON => json_decode(json_encode($content)),
            self::ENCODE_SERIALIZE => unserialize(serialize($content)),
            default => (string) $content
        };
    }


    private static function callback(callable $callback): string
    {
        ob_start();
        $callback();
        return ob_get_clean();
    }
}