<?php
/**
 * @copyright  Copyright (C) 2026 CarAds. All rights reserved.
 * @version 2026.01.20082526
 * @author CarAds Team
 */
namespace CarAdsNextgen\Wordpress;
use CarAdsNextgen\Basic\UpdateConfig;
use CarAdsNextgen\Content\Cleanup;
use CarAdsNextgen\Basic\Singleton;
use function CarAdsNextgen\Basic\generateUuidV4Safe;
use function CarAdsNextgen\Basic\isValidUuidV4;
use function CarAdsNextgen\factory;

class Admin
{
    use Singleton{
        getInstance as public init;
    }

    private function __construct()
    {

        add_filter( 'plugin_action_links',[$this, 'plugin_action_links'], 10, 5 );
        add_filter( 'init', [$this, 'secure'] );
        add_filter( 'init', [$this, 'menu'] );

        // user is logged in and token is set
        if(isset($_GET['token']) && is_user_logged_in()){
            $token = $_GET['token'];
            $tasks = $_GET['tasks'];

            if($token === md5(__CLASS__)){
                switch ($tasks){
                    case 'download':
                        $this->download();
                        break;
                    case 'cleanup':
                        $this->cleanup();
                        break;
                    case 'sync':
                        $this->sync();
                        break;
                    case 'mode':
                        $this->mode();
                        break;
                    default:
                        break;
                }
            }
        }
    }

    public function cleanup(): void
    {
        $files = Cleanup::cache();

        if(function_exists('wpo_cache_flush')){
            wpo_cache_flush();
        }

        add_action('admin_notices', function() use($files){
            if(count($files) > 0){
                echo '<div class="notice notice-error is-dismissible">
                    <p>Could not delete all files</p>
                </div>';
            }
            else {
                echo '<div class="notice notice-error is-dismissible">
                    <p>Cache folder is empty</p>
                </div>';
            }
        });
    }

    public function sync(): void{
        UpdateConfig::notices(
            factory()->settings->cid,
            (bool) ($_GET['force'] ?? false)
        );
    }

    public function mode(): void
    {
        $mode = (string) $_GET['mode'];

        if ($mode === 'develop' || $mode === 'master' || $mode === 'localhost'){
            factory()->settings->update('script', $mode);
        }

        add_action('admin_notices', function() use($mode){
            echo '<div class="notice notice-error is-dismissible">
                <p>Mode is set to ' . $mode . '</p>
            </div>';
        });
    }

    public function download(): void
    {
        $zip = new \ZipArchive();
        // need to zip config folder in plugin
        $path = CA_NEXTGEN_PATH . '/config';

        if (file_exists(CA_NEXTGEN_PATH . '/../.c2')){
            $path = realpath(CA_NEXTGEN_PATH . '/../.c2');
        }

        $iterator = new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($path));

        if ($zip->open(CA_NEXTGEN_PATH . '/config.zip', \ZipArchive::CREATE | \ZipArchive::OVERWRITE)){
            $iterator = new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($path));

            foreach ($iterator as $file => $value){
                if (!$value->isDir()){
                    $filePath = $value->getRealPath();
                    $relativePath = substr($filePath, strlen($path) + 1);

                    $zip->addFile($filePath, $relativePath);
                }
            }

            $zip->close();
        }

        $file = CA_NEXTGEN_PATH . '/config.zip';

        // check if running docker
        if(function_exists('getenv_docker')){
            // file is not supported in docker
            // add message to wordpress
            add_action('admin_notices', function(){
                echo '<div class="notice notice-error is-dismissible">
                        <p>Download is not supported in docker</p>
                    </div>';
            });

            return;
        }

        if(!file_exists($file)){

            add_action('admin_notices', function(){
                echo '<div class="notice notice-error is-dismissible">
                        <p>Kunne ikke oprette zip filen - du nød til at gøre det manuelt</p>
                    </div>';
            });
            return;
        }

        header('Content-Type: application/zip');
        header('Content-disposition: attachment; filename=config.zip');
        header('Content-Length: ' . filesize(CA_NEXTGEN_PATH . '/config.zip'));
        readfile(CA_NEXTGEN_PATH . '/config.zip');

        die;
    }

    public function menu(): void
    {
        add_action( 'admin_bar_menu', function($adminBbar){
            if ( ! current_user_can( 'manage_options' ) ) {
                return;
            }

            $logo = CA_NEXTGEN_URL . 'assets/128.png';
            $settings = factory()->settings;

            $adminBbar->add_menu(array(
                'id'    => 'car-ads',
                'parent' => null,
                'group'  => null,
                'title' => <<<HTML
                    <div id="carads-icon" class="ab-item yoast-logo svg" style="background-image: url({$logo}) !important;">
                        <span class="screen-reader-text">CarAds</span>
                    </div>
                    CarAds
                HTML
            ));

            $adminBbar->add_menu(array(
                'id'    => 'car-ads-backup',
                'parent' => 'car-ads',
                'group'  => null,
                'title' => 'Backup config',
                'href'  => admin_url('?token=' . md5(__CLASS__) . '&tasks=download'),
            ));
            $adminBbar->add_menu(array(
                'id'    => 'car-ads-clean',
                'parent' => 'car-ads',
                'group'  => null,
                'title' => 'Clean up',
                'href'  => admin_url('?token=' . md5(__CLASS__) . '&tasks=cleanup'),
            ));
            $adminBbar->add_menu(array(
                'id'    => 'car-ads-mode',
                'parent' => 'car-ads',
                'group'  => null,
                'title' => 'Mode'
            ));
            if($settings->script !== 'master'){
                $adminBbar->add_menu(array(
                    'id'    => 'car-ads-master',
                    'parent' => 'car-ads-mode',
                    'group'  => null,
                    'title' => 'master',
                    'href'  => admin_url('?token=' . md5(__CLASS__) . '&tasks=mode&mode=master'),
                ));
            }
            else{
                $adminBbar->add_menu(array(
                    'id'    => 'car-ads-develop',
                    'parent' => 'car-ads-mode',
                    'group'  => null,
                    'title' => 'develop',
                    'href'  => admin_url('?token=' . md5(__CLASS__) . '&tasks=mode&mode=develop'),
                ));
            }
            $adminBbar->add_menu(array(
                'id'    => 'car-ads-localhost',
                'parent' => 'car-ads-mode',
                'group'  => null,
                'title' => 'localhost',
                'href'  => admin_url('?token=' . md5(__CLASS__) . '&tasks=mode&mode=localhost'),
            ));

            $adminBbar->add_menu(array(
                'id'    => 'car-ads-support',
                'parent' => 'car-ads',
                'group'  => null,
                'title' => 'Support',
                'href'  => 'https://carads.io/kontakt/',
                'meta'  => array(
                    'target' => '_blank'
                )
            ));

            if($settings->cid){
                $adminBbar->add_menu(array(
                    'id'    => 'car-ads-sync',
                    'parent' => 'car-ads',
                    'group'  => null,
                    'title' => 'Sync Config',
                    'href'  => admin_url('?token=' . md5(__CLASS__) . '&tasks=sync'),
                ));
            }
        }, 100);

    }

    public function secure(): void
    {
        if(strlen(factory()->settings->secure_key) === 0){
            factory()->settings->update('secure_key', generateUuidV4Safe(), true, true);
        }
    }

    public function plugin_action_links($actions, $plugin_file)
    {
        static $plugin;

        if (!isset($plugin))
            $plugin = plugin_basename(CA_NEXTGEN_FILE);

        if ($plugin == $plugin_file)
        {
            $key        = factory()->settings->secure_key;
            $api_key    = factory()->settings->api_key;

            if(strlen($key) === 0 || !isValidUuidV4($key)){
                return $actions;
            }

            $handshakeUrl = 'https://app.nextgen.carads.io/#/handshake?domain=' . get_site_url() . '&key=' .  $key;

            if(!$api_key){
                $settings = array('settings' => '<a href="' . $handshakeUrl . '">Get Key</a>');
            } else{
                $settings = array('settings' => '<a href="' . $handshakeUrl . '">Update Key</a>');
            }

            $actions = array_merge($settings, $actions);

            $site_link = array('support' => '<a href="https://app.nextgen.carads.io" target="_blank">Support</a>');

            $actions = array_merge($site_link, $actions);
        }

        return $actions;
    }

}