<?php
/**
 * @copyright  Copyright (C) 2026 CarAds. All rights reserved.
 * @version 2026.01.20082526
 * @author CarAds Team
 */
namespace CarAdsNextgen\Wordpress;
use CarAdsNextgen\Customize\Checkbox;
use CarAdsNextgen\Customize\Color;
use CarAdsNextgen\Customize\Section;
use CarAdsNextgen\Customize\Select;
use CarAdsNextgen\Customize\Text;
use CarAdsNextgen\Utils\Cache;
use CarAdsNextgen\Basic\Singleton;
use WP_Customize_Manager;
use WP_Customize_Panel;
use WP_Customize_Section;
use function CarAdsNextgen\Basic\render;
use function CarAdsNextgen\Basic\rgb;

class Customize
{
    use Singleton;

    const string CSS_FILE = CA_NEXTGEN_PATH . '/style.css';

    private function __construct()
    {
        add_action('customize_register', [$this, 'register'], 1, 99);
        add_action('customize_save_after', [$this, 'save'], 1, 99);
        add_action('customize_preview_init', function($customize){
            add_action('wp_footer', function() use($customize){
                $this->outputCSS($customize);
            });
        });
    }

    public function outputCSS(WP_Customize_Manager $customize): void
    {
        echo '<style id="customize_carads_css">';
        echo $this->renderCss($customize);
        echo '</style>';
    }

    public function save(WP_Customize_Manager $customize) : void
    {
        @file_put_contents(self::CSS_FILE, $this->renderCss($customize));
    }


    private function renderCss(WP_Customize_Manager $customize): string
    {
        return render(CA_NEXTGEN_PATH . '/template-parts/css.php', ['customize' => $customize]);
    }

    private function panel(WP_Customize_Manager $customize) : WP_Customize_Panel{
        return $customize->add_panel( 'c2_settings', array(
            'title' => __( 'CarAds Indstillinger', 'CarAdsLight' ),
            'description' => '',
            'priority' => 30,
            'capability' => 'manage_options'
        ));
    }

    public function register(WP_Customize_Manager $customize) : void{
        $panel = $this->panel($customize);
        $this->general($customize, $panel);
        $this->css($customize, $panel);
        $this->cache($customize, $panel);
    }

    private function cache(WP_Customize_Manager $customize, WP_Customize_Panel $panel): void
    {
        $section = $customize->add_section(
            new Section(
                __( 'Cache', 'CarAdsLight' ),
                $panel->id,
                $customize
            )
        );

        // remove all seo tags from 3rd party
        (new Checkbox())
            ->key('CarAdsNextgenRemoveAllSeoTagsFrom3rdParty')
            ->label(__( 'Remove all seo tags from 3rd party', 'CarAdsNextgen'))
            ->default(false)
            ->description('virker men er kun til S.O.S - Da den fjerne alle defineret seo, (ved replace string) - så kan det være at det ikke virker som forventet - også kun til S.O.S')
            ->section($section)
            ->register($customize);
    }

    private function general(WP_Customize_Manager $customize, WP_Customize_Panel $panel): void
    {
        $section = $customize->add_section(
            new Section(
                __( 'Generalt', 'CarAdsLight' ),
                $panel->id,
                $customize
            )
        );

        $releases = Cache::byReturn(function(){
            $data = $this->fetch();
            $release = [];
            $labels = [
                'no_updates' => 'Ingen opdateringer',
                'updates_week' => 'Opdateres jævnligt + hotfixes, patches og nye features - anbefales',
                'updates_any' => 'Opdateres løbende + hotfixes, patches og nye features + ikke testet ting',
            ];

            foreach($data as $key => $value){
                $label = $value['base'] . ' - ' . $value['version'] . ' - ' . $labels[$value['live']];

                if($value['live'] === 'no_updates'){
                    $label .= ' - ' . $value['time'] . ' **';
                }
                else if($value['live'] === 'updates_week'){
                    $label .= ' + hotfixes, patches og nye features';
                }

                $release[$value['path']] = $label;
            }

            return $release;
        }, 'release_notes.json', (60*5), Cache::ENCODE_JSON);


        $releases['local'] = 'Lokal udvikling';

        (new Select())
            ->key('CarAdsNextgenScript')
            ->default('master')
            ->label(__( 'script', 'CarAdsNextgen'))
            ->choices(...$releases)
            ->section($section)
            ->register($customize);

        (new Text())
            ->key('CarAdsNextgenClientID')
            ->label(__( 'ClientID', 'CarAdsNextgen'))
            ->placeholder('-1')
            ->default('-1')
            ->description('not created yet')
            ->section($section)
            ->register($customize);

        (new Text())
            ->key('CarAdsNextgenApiKey')
            ->label(__( 'API Key', 'CarAdsNextgen'))
            ->default('')
            ->description('API Key - hvis du ikke har en, så kontakt CarAdsNextgen support')
            ->section($section)
            ->register($customize);
        (new Text())
            ->key('CarAdsNextgenSecureKey')
            ->label(__( 'Secure Key', 'CarAdsNextgen'))
            ->default('')
            ->description('Secure Key - hvis du ikke har en, så kontakt CarAdsNextgen support')
            ->section($section)
            ->register($customize);

        (new Checkbox())
            ->key('CarAdsNextgenShadowRoot')
            ->label(__( 'Shadow Root', 'CarAdsNextgen'))
            ->default(true)
            ->description('Enable Shadow Root - this is experimental and not fully tested yet')
            ->section($section)
            ->register($customize);

        // test ip
        (new Text())
            ->key('CarAdsNextgenTestIP')
            ->label(__( 'Test IP', 'CarAdsNextgen'))
            ->default('')
            ->description('Test IP - hvis flere udvikler på samme tid, kan man benytte , for at adskille dem')
            ->section($section)
            ->register($customize);


        // enable carlead tracking
        (new Checkbox())
            ->key('CarAdsNextgenCarLeadTracking')
            ->label(__( 'CarLead Tracking', 'CarAdsNextgen'))
            ->default(false)
            ->description('Enable CarLead Tracking')
            ->section($section)
            ->register($customize);

        // enable beta version
        (new Checkbox())
            ->key('CarAdsNextgenBetaChannel')
            ->label(__( 'CarAds Wordpress Update Channel', 'CarAdsNextgen'))
            ->default(false)
            ->description('Dette er en beta version af wordpress plugin - så når dette slået til vil du modtage beta versioner og dette frarådes til produktion')
            ->section($section)
            ->register($customize);
        (new Checkbox())
            ->key('CarAdsNextgenOptimize')
            ->label(__( 'Optimize css load for single components when it need', 'CarAdsNextgen'))
            ->default(false)
            ->description('Optimize')
            ->section($section)
            ->register($customize);

        // select with options about form layout you wish to use
        $options = [
            'old-school' => 'Old School',
            'floating' => 'Floating Labels',
            'floating-outlined' => 'Floating Outlined',
            'all' => 'All',
        ];

        (new Select())
            ->key('CarAdsNextgenFormLayout')
            ->label(__( 'Form Layout', 'CarAdsNextgen'))
            ->default('floating-labels')
            ->choices(...$options)
            ->section($section)
            ->register($customize);

        (new Checkbox())
            ->key('CarAdsNextgenAllow3rdPartyLogin')
            ->label(__( 'Allow 3rd party login', 'CarAdsNextgen'))
            ->default(false)
            ->description('Allow 3rd party login')
            ->section($section)
            ->register($customize);
    }
    private function css(WP_Customize_Manager $customize, WP_Customize_Panel $panel): void
    {
        $section = $customize->add_section(
            new Section(
                __( 'Default CSS', 'CarAdsLight' ),
                $panel->id,
                $customize
            )
        );

        $color = [
            'gray_1' => rgb(243,244,246),
            'gray_2' => rgb(229,231,235),
            'gray_3' => rgb(209,213,219),
            'gray_4' => rgb(156,163,175),
            'gray_5' => rgb(107,114,128),
            'gray_6' => rgb(75,85,99),
            'gray_7' => rgb(55,65,81),
            'gray_8' => rgb(31,41,55),
            'gray_9' => rgb(17,24,39),
            'stone_1' => rgb(245,245,244),
            'stone_2' => rgb(231,229,228),
            'stone_3' => rgb(214,211,209),
            'stone_4' => rgb(168,162,158),
            'stone_5' => rgb(120,113,108),
            'stone_6' => rgb(87,83,78),
            'stone_7' => rgb(68,64,60),
            'stone_8' => rgb(41,37,36),
            'stone_9' => rgb(28,25,23),
        ];

        foreach($color as $key => $value){
            $this->color('CarAdsNextgenCSS_Color_' . ucfirst($key), $value, ucfirst($key), $section, $customize);
        }

        $section = $customize->add_section(
            new Section(
                __( 'Santander CSS', 'CarAdsLight' ),
                $panel->id,
                $customize
            )
        );
        $this->color('CarAdsNextgenCSS_santander_body_input_background', '', 'Link', $section, $customize);
        $this->color('CarAdsNextgenCSS_santander_body_background', '', 'Link', $section, $customize);
        $this->color('CarAdsNextgenCSS_santander_header_background', '', 'Link', $section, $customize);
        $this->color('CarAdsNextgenCSS_santander_footer_button_background', '', 'Link', $section, $customize);
        $this->color('CarAdsNextgenCSS_santander_footer_button_border_color', '', 'Link', $section, $customize);
        $this->color('CarAdsNextgenCSS_santander_footer_button_color', '', 'Link', $section, $customize);
    }

    private function color(string $name, string $value, string $label, WP_Customize_Section $section, WP_Customize_Manager $customize): void
    {
        (new Color())
            ->key($name)
            ->label($label)
            ->default($value)
            ->section($section)
            ->register($customize);
    }


    private function fetch(){
        $url = "https://release.js.nextgen.carads.io/";

        $data = (new \WP_Http())->get($url);

        return json_decode($data['body'] ?? '[]', true);
    }
}