<?php
/**
 * @copyright  Copyright (C) 2026 CarAds. All rights reserved.
 * @version 2026.01.20082526
 * @author CarAds Team
 */
namespace CarAdsNextgen\Wordpress;
use CarAdsNextgen\Controller\DealerController;
use CarAdsNextgen\Controller\DealersController;
use CarAdsNextgen\Controller\ProductController;
use CarAdsNextgen\Controller\ProductsController;
use CarAdsNextgen\Controller\ProductsDiffController;
use CarAdsNextgen\Controller\SiteMapController;
use CarAdsNextgen\Factory;
use CarAdsNextgen\Basic\Seo;
use CarAdsNextgen\Basic\Singleton;
use function CarAdsNextgen\Basic\css_inject_helper;
use function CarAdsNextgen\Basic\elementor_reload_preview;
use function CarAdsNextgen\Basic\simple_cdn_version;
use function CarAdsNextgen\factory;

/**
 * Class Rewrite
 * @package CarAdsNextgen
 * @method static self getInstance
 */
class Rewrite
{
    private ?string $version = null;
    private ?string $time = null;
    private bool|null $optimized_assets = null;

	use Singleton{
	    getInstance as public init;
    }


    const array CONTROLLER = [
        'product-detail'            => ProductController::class,
        'product-list'              => ProductsController::class,
        'product-list-filter'       => ProductsController::class,
        'product-diff'              => ProductsDiffController::class,
        'sitemap'                   => SiteMapController::class,
    ];

    public function __construct()
    {
        $this->time = date('YmdH');
        $route      = factory()->route;

        add_action('init', function () use($route){
            $route->register_rewrites();

            if($route->active())
            {
                $this->render();
            }
        });

        add_filter('wp_headers', [$this, 'wpHeaders'], 11, 1);
        add_filter('wp_head', [$this, 'meta'], 11, 1);
        add_filter('wp_enqueue_scripts', [$this, 'css'], 99);
        add_filter('wp_script_attributes', function (array $data) {
            if($data['id'] === 'carads-main-js-module'){
                // https://developer.mozilla.org/en-US/docs/Web/HTML/Reference/Attributes/crossorigin
                $data['crossorigin'] = 'anonymous';
            }

            return $data;
        }, 10, 2);
        add_filter('elementor/preview/enqueue_scripts', function(){
            elementor_reload_preview();
        }, 999);
    }

    /**
     * @deprecated use cdnVersion instead
     * will be removed in future version
     * @todo remove this function in future version
     */
    public function getVersion() : string{
        return simple_cdn_version();
    }

    public function cssFile() : ?string{
        return css_inject_helper((Factory::getInstance()->settings->optimize ? 'optimize.css' : 'style.css'));
    }

    public function jsFile() : string{
        $version = simple_cdn_version();
        $query = http_build_query($this->mainJsQuery());

        return (CA_TEST || ($version === 'localhost' || $version === 'local')) ?
            ("http://localhost:3000/src/main.js?{$query}") :
            ("https://js.nextgen.carads.io/{$version}/assets/main.js?{$query}");
    }

    private function mainJsQuery() : array {
        $settings = Factory::getInstance()->settings;

        return [
            'v' => $this->time,
            'wx' => CA_NEXTGEN_VERSION,
            'environment' => $settings->cid,
            'shadow_root' => $settings->shadow_root ? 1 : 0,
        ];
    }

    /**
     * @deprecated use jsFile and cssFile instead
     * will be removed in future version
     * @todo remove this function in future version
     */
    public function jsDealer() : string{
        return "";
    }

    /**
     * @deprecated use jsFile and cssFile instead
     * will be removed in future version
     * @todo remove this function in future version
     */
    public function cssDealer() : string{
        return "";
    }

    public function css() : void{
        $file = $this->cssFile();

        wp_enqueue_style('carads-main', $file, [], null);

        $settings = Factory::getInstance()->settings;

        if ($settings->optimize){
            wp_enqueue_style('car-ads-icons', css_inject_helper('icons.css'), [], null);
            wp_enqueue_style('car-ads-opening-times', css_inject_helper('department/opening_times/basic.css'));

            /**
             * Before aksing why id is not same.
             * - virtual config settings cache will not reload php - so for javascript can use correct css,
             * id is make this way to make sure it is unique and we only need to add not update css file by id
             */
            if ($settings->form_layout === 'floating'){
                wp_enqueue_style('car-ads-form-floating', css_inject_helper('forms/floating-form.css'), [], null);
            }
            else if($settings->form_layout === 'floating-outlined'){
                wp_enqueue_style('car-ads-form-floating-outlined', css_inject_helper('forms/floating-outlined-form.css'), [], null);
            }
            else if($settings->form_layout === 'old-school'){
                wp_enqueue_style('car-ads-form-old', css_inject_helper('forms/old-school.css'), [], null);
            }
            else {
                wp_enqueue_style('car-ads-form-all', css_inject_helper('forms/_form.css'), [], null);
            }
        }

        $src = $this->jsFile(); // must be a public URL, not a filesystem path

        wp_register_script_module(
            'carads-main',
            $src,
            []
        );

        wp_enqueue_script_module('carads-main');
        if (Factory::getInstance()->settings->carlead_enabled) {
            wp_enqueue_script('car-ads-carlead', 'https://nextgen.carads.io/assets/js/carlead.min.js?v=' . time() . '&wx=' . CA_NEXTGEN_VERSION);
        }
    }

    public function meta(): void
    {
        echo '<link rel="dns-prefetch" href="https://nextgen.carads.io" />';
        echo '<link rel="dns-prefetch" href="https://js.nextgen.carads.io" />';
        echo '<link rel="preconnect" href="https://nextgen.carads.io" />';
        echo '<link rel="preconnect" href="https://js.nextgen.carads.io" />';
    }

    /**
     * After we move validate from ajax to rest api, not sure we need this other then for iframe calling..
     * @TODO sure remember test if we need this or not
     */
    public function wpHeaders($headers)
    {
        $domains = ['https://app.nextgen.carads.io', 'https://nextgen.carads.io'];
        // host without port number
        $host = explode(':', $_SERVER['HTTP_HOST'])[0];

        if ($host === 'localhost' && CA_TEST)
        {
            $domains[] = "{$_SERVER['REQUEST_SCHEME']}://{$_SERVER['HTTP_HOST']}";
            $domains[] = "{$_SERVER['REQUEST_SCHEME']}://localhost:3000";
        }

        /** allow websites to connect to website her  */
        $headers['Access-Control-Allow-Origin']  = implode(', ', $domains);
        $headers['Access-Control-Allow-Methods'] = 'GET, POST, OPTIONS';
        $headers['Access-Control-Allow-Headers'] = 'Authorization, Content-Type, Accept';
        /** make header for load javascript faster */
        $headers['link'] = '<https://app.nextgen.carads.io>; rel=preconnect; crossorigin';

        return $headers;
    }

    protected function render() : void
    {
        $page = factory()->route->active();
        $this->optimizeTemplateRedirect();
        if(self::CONTROLLER[$page->controller] ?? null)
        {
            $seo = new Seo([
                '3rd_party_seo_active' => false,
            ]);

            add_action('template_redirect', function() use($page, $seo){
                $controller = self::CONTROLLER[$page->controller];
                (new $controller($seo, factory()->route))->render();
            }, 99);
        }
    }

    protected function optimizeTemplateRedirect() : void
    {
        /**
         * Hack to remove redirect_canonical and wp_old_slug_redirect so we can take 100% control of url.
         */
        add_action('template_redirect', function(){
            /**  */
            global $wp_filter, $wp_filters;

            $tmp = $wp_filter['template_redirect'][10];
            foreach($tmp AS $key => $value){
                if($value['function'] === 'wp_old_slug_redirect'){
                    unset($tmp[$key]);
                }
                if($value['function'] === 'redirect_canonical'){
                    unset($tmp[$key]);
                }
            }

            $wp_filter['template_redirect'][10] = $tmp;
        }, 9);
    }
}