<?php
/**
 * @copyright  Copyright (C) 2026 CarAds. All rights reserved.
 * @version   2026.01.20082528
 * @author    CarAds Team
 */
namespace CarAdsLite\Basic;

use Exception;

/**
 * Class Factory
 *
 * @property-read Configs $configs
 * @property-read bool $wpOptimize
 * @property-read bool $wp_optimize - alias for $wpOptimize
 * @property-read bool $yoastSeo
 * @property-read bool $yoast_seo - alias for $yoastSeo
 * @property-read bool $yoastSeoPro
 * @property-read bool $yoast_seo_pro - alias for $yoastSeoPro
 * @property-read BasicSettings $settings
 * @property-read bool $is_test_ip - always false in this default factory
 * @property-read string $url
 */
abstract class DefaultFactory
{
    /**
     * @throws Exception
     */
    public function __set(string $name, $value): void
    {
        throw new Exception("Cannot set read-only property {$name} in " . __CLASS__);
    }

    /**
     * @throws Exception
     */
    public function __get(string $name): mixed{
        return match ($name) {
            'wpOptimize', 'wp_optimize' => $this->wp_optimize(),
            'yoastSeo', 'yoast_seo' => $this->yoast_exists(),
            'yoastSeoPro', 'yoast_seo_pro' => $this->yoast_pro_exists(),
            'settings' => $this->settings(),
            'url' => $this->url(),
            'is_test_ip' => false,
            default => throw new Exception("Property {$name} does not exist in " . __CLASS__),
        };
    }

    private function wp_optimize(): bool
    {
        return (
            is_plugin_active('wp-optimize-premium/wp-optimize-premium.php') ||
            is_plugin_active('wp-optimize/wp-optimize.php')
        );
    }

    private function yoast_exists(): bool
    {
        return is_plugin_active('wordpress-seo/wp-seo.php');
    }

    private function yoast_pro_exists(): bool
    {
        return is_plugin_active('wordpress-seo-premium/wp-seo-premium.php');
    }

    private function url(): string
    {
        static $url = null;
        if($url === null){
            $url = rtrim(parse_url($_SERVER['REQUEST_URI'] ?? '', PHP_URL_PATH), '/');
        }
        return $url;
    }

    private function settings(): BasicSettings{}

    /**
     * @throws Exception
     */
    public function __call(string $name, array $arguments)
    {
        throw new Exception("Method {$name} does not exist in " . __CLASS__);
    }
}