<?php
/**
 * @copyright  Copyright (C) 2026 CarAds. All rights reserved.
 * @version   2026.01.20082526
 * @author    CarAds Team
 */
namespace CarAds404\Basic;
use WP_Customize_Manager;
use WP_Customize_Panel;
use WP_Customize_Section;

/** @noinspection DuplicatedCode */
class CustomizeHelper
{
    const string SANITIZE_EMPTY_STRING = 'empty_string';
    const string SANITIZE_BOOL = 'bool';

    const string MEDIA_MIME_IMAGE = 'image';
    const string MEDIA_MIME_VIDEO = 'video';
    const string MEDIA_MIME_AUDIO = 'audio';


    /** @noinspection DuplicatedCode */
    static public function Text(
        WP_Customize_Manager $customize,
        string $key = '',
        string $label = '',
        string $default = '',
        string $placeholder = '',
        string $section = '',
        string $description = '',
        string $transport = 'refresh',
        string|callable|null $sanitize_callback = null
    ): void
    {
        $settings =  [
            'default'           => $default,
            'type'              => 'option',
            'transport'         => $transport,
            'theme_supports'    => '',
            'capability'        => 'manage_options',
        ];

        if(is_string($sanitize_callback)){
            $settings['sanitize_callback'] = self::sanitize_callback($sanitize_callback, $default);
        } elseif(is_callable($sanitize_callback)){
            $settings['sanitize_callback'] = $sanitize_callback;
        }

        $customize->add_setting($key,$settings);

        $customize->add_control($key, [
            'label'   => $label,
            'description' => $description,
            'section' => $section,
            'type' => 'text',
            'input_attrs' => array(
                'placeholder' => $placeholder
            ),
            'settings'   => $key
        ]);
    }

    /** @noinspection DuplicatedCode */
    static public function Select(
        WP_Customize_Manager $customize,
        string $key = '',
        string $label = '',
        string $default = '',
        array $choices = [],
        string $section = '',
        string $description = '',
        string $transport = 'refresh'
    ): void
    {
        $settings = [
            'default'           => $default,
            'type'              => 'option',
            'transport'         => $transport,
            'capability'        => 'manage_options',
        ];

        $customize->add_setting($key, $settings);

        $customize->add_control($key, [
            'label'   => $label,
            'description' => $description,
            'section' => $section,
            'type' => 'select',
            'settings'   => $key,
            'choices' => $choices
        ]);
    }

    /** @noinspection DuplicatedCode */
    static public function Section($title, $panel, $customize): WP_Customize_Section
    {
        static $index;

        if(!isset($index)){
            $index = 0;
        }

        return $customize->add_section(
            new WP_Customize_Section(
                $customize,
                __CLASS__ . '_' . (++$index),
                [
                    'title' => $title,
                    'panel' => $panel,
                    'description' => '',
                    'capability' => 'manage_options'
                ]
            )
        );
    }

    /** @noinspection DuplicatedCode */
    static public function panel(
        WP_Customize_Manager $customize,
        string $key,
        string $title,
        string $description = '',
        int $priority = 30
    ) : WP_Customize_Panel
    {
        return $customize->add_panel( $key, array(
            'title' => $title,
            'description' => $description,
            'priority' => $priority,
            'capability' => 'manage_options'
        ));
    }

    /** @noinspection DuplicatedCode */
    static public function Checkbox(
        WP_Customize_Manager $customize,
        string $key = '',
        string $label = '',
        bool $default = false,
        string $section = '',
        string $description = '',
        string $transport = 'refresh'
    ): void
    {
        $customize->add_setting($key, [
            'default'           => $default,
            'type'              => 'option',
            'transport'         => $transport,
            'capability'        => 'manage_options',
            'sanitize_callback' => function($checked){
                return $checked ? '1' : '0';
            }
        ]);

        $customize->add_control($key, [
            'label'   => $label,
            'description' => $description,
            'section' => $section,
            'type' => 'checkbox',
            'settings'   => $key
        ]);
    }

    static public function Editor(
        WP_Customize_Manager $customize,
        string $key = '',
        string $label = '',
        string $default = '',
        string $section = '',
        string $description = '',
        string $transport = 'refresh',
        string|callable|null $sanitize_callback = null
    ): void
    {
        $settings = [
            'default'           => $default,
            'type'              => 'option',
            'transport'         => $transport,
            'theme_supports'    => '',
            'capability'        => 'manage_options',
            'sanitize_callback' => 'wp_kses_post'
        ];

        $customize->add_setting($key, $settings);

        $customize->add_control( new \WP_Customize_Code_Editor_Control( $customize, $key, [
            'label'   => $label,
            'description' => $description,
            'section' => $section,
            'settings'   => $key
        ]));
    }

    static public function Media(
        WP_Customize_Manager $customize,
        string $key = '',
        string $label = '',
        string $section = '',
        string $description = '',
        string $transport = 'refresh',
        string|array $mime_type = self::MEDIA_MIME_IMAGE
    ): void{
        $customize->add_setting( $key, [
            'default'   => '',
            'transport' => 'refresh',
            'type'      => 'option',
            'capability'=> 'manage_options'
        ]);
        $customize->add_control( new \WP_Customize_Media_Control( $customize, $key, [
            'label'       => $label,
            'description' => $description,
            'section'     => $section,
            'settings'    => $key,
            'mime_type'   => $mime_type, // image, video, audio
        ]));
    }

    static public function Textarea(
        WP_Customize_Manager $customize,
        string $key = '',
        string $label = '',
        string $default = '',
        string $section = '',
        string $description = '',
        string $transport = 'refresh',
        string|callable|null $sanitize_callback = null
    ): void
    {
        $settings = [
            'default'           => $default,
            'type'              => 'option',
            'transport'         => $transport,
            'theme_supports'    => '',
            'capability'        => 'manage_options',
        ];

        if(is_string($sanitize_callback)){
            $settings['sanitize_callback'] = self::sanitize_callback($sanitize_callback, $default);
        } elseif(is_callable($sanitize_callback)){
            $settings['sanitize_callback'] = $sanitize_callback;
        }

        $customize->add_setting($key, $settings);

        $customize->add_control($key, [
            'label'   => $label,
            'description' => $description,
            'section' => $section,
            'type' => 'textarea',
            'settings'   => $key
        ]);
    }

    static private function sanitize_callback(string $sanitize, mixed $default = null): ?callable{
        return match ($sanitize){
            'empty_string' => function($value) use($default){
                return $value === '' ? $default : (string)$value;
            },
            'bool' => function($checked){
                // https://wordpress.stackexchange.com/questions/68052/adding-a-checkbox-to-the-theme-customizer
                return $checked ? '1' : '0'; // it hack for get bool to work anytimes
            },
            default => null
        };
    }
}