<?php
/**
 * @copyright  Copyright (C) 2026 CarAds. All rights reserved.
 * @version 2026.01.20082526
 * @author CarAds Team
 */
namespace CarAds404;
use stdClass;

/** @noinspection DuplicatedCode */
function simpleUpdater(string $url, string $version = '0.0.0', $base = null, array $icons = null): object
{
    return new class($url, $version, $base, $icons) {
        private string $url;

        private string $slug;
        private string $version;

        private string $base;

        private ?array $icons = null;

        /** @noinspection DuplicatedCode */
        public function __construct(
            string $url, string
                   $version = '0.0.0',
                   $base = null,
                   $icons = null
        )
        {
            $this->url      = $url;
            $this->base     = $base;
            $this->version  = $version;
            $this->slug     = dirname($this->base);
            $this->icons    = $icons ? (array) $icons : null;

            add_filter('plugins_api', [$this, 'update'], 10, 3);
            add_filter('pre_set_site_transient_update_plugins', [$this, 'pre_set_site_transient_update_plugins']);
        }

        /** @noinspection DuplicatedCode */
        private function fetch() : ?array {
            $response = wp_remote_get($this->url, ['timeout' => 5]);
            if (is_wp_error($response)) {
                return null;
            }

            $meta = json_decode(wp_remote_retrieve_body($response), true);
            if (!$meta){
                return null;
            }

            return $meta;
        }

        /** @noinspection DuplicatedCode */
        public function update(mixed $result, string $action, \stdClass $args ) {
            if ($action !== 'plugin_information' || empty($args->slug) || $args->slug !== $this->slug) {
                return $result;
            }

            $meta = $this->fetch();

            if (!$meta) {
                return $result;
            }

            $info                   = new \stdClass();
            $info->name             = $meta['name'] ?? 'My Plugin';
            $info->slug             = $this->slug;
            $info->version          = $meta['version'] ?? $this->version;
            $info->author           = $meta['author'] ?? 'CarAds.io';
            $info->homepage         = $meta['homepage'] ?? 'carads.io';
            $info->requires         = $meta['requires'] ?? '';
            $info->tested           = $meta['tested'] ?? '';
            $info->sections         = $meta['sections'] ?? ['description' => $meta['description'] ?? ''];
            $info->download_link    = $meta['download_url'] ?? '';
            $info->timestamp        = isset($meta['timestamp']) ? strtotime($meta['timestamp']) : time();

            return $info;
        }

        /** @noinspection DuplicatedCode */
        public function pre_set_site_transient_update_plugins(mixed $transient) : \stdClass
        {
            if (!is_object($transient))
            {
                $transient = new \stdClass();
            }

            $meta = $this->fetch();

            if (!$meta || empty($meta['version'])) {
                return $transient;
            }

            $obj = (object)[
                'slug'          => $this->slug,
                'plugin'        => $this->base,
                'new_version'   => $meta['version'],
                'url'           => $meta['homepage'] ?? null,
                'package'       => $meta['download_url'] ?? null,
            ];

            if($this->icons){
                $obj->icons = $this->icons;
            }

            $transient->checked[$this->base] = $meta['version'];

            if (version_compare($this->version, $meta['version'], '<')) {
                if(!isset($transient->response)){
                    $transient->response = [];
                }

                if(isset($data->no_update) && is_array($data->no_update) && array_key_exists($this->base, $data->no_update)){
                    unset($data->no_update[$this->base]);
                }

                $transient->response[$this->base] = $obj;
            } else {
                if (isset($transient->response[$this->base])) {
                    unset($transient->response[$this->base]);
                }

                if(!isset($transient->no_update)){
                    $transient->no_update = [];
                }

                $transient->no_update[$this->base] = $obj;
            }

            return $transient;
        }
    };
}

function factory(): Factory
{
    return Factory::getInstance();
}

/** @noinspection DuplicatedCode */
function render(string $file, array|stdClass $data) : string
{
    extract((array) $data);

    ob_start();
    include func_get_arg(0);

    return ob_get_clean();
}