<?php
/**
 * @copyright  Copyright (C) 2026 CarAds. All rights reserved.
 * @version 2026.01.20082523
 * @author CarAds Team
 */
namespace CarAdsAuth\User;
use CarAdsAuth\Utils\Singleton;
use JetBrains\PhpStorm\NoReturn;
use function CarAdsAuth\isCarAdsUser;

class Authenticate
{
    const string TRANSIENT_KEY = 'carads_auth_platform_login_';

    use Singleton{
        getInstance as public init;
    }

    private bool $isError = false;
    private bool $isOtpError = false;

    private function __construct()
    {
        add_filter('authenticate', [$this, 'platform_login'], 0, 3);
        add_filter('authenticate', [$this, 'after_login'], 999, 3);
        add_filter('login_form_two_step', [$this, 'token_view']);
        add_filter('login_form_other', [$this, 'token_view']);

        if (CA_AUTH_TEST){
            add_filter('phpmailer_init', [$this, 'phpmailer_init']);
            add_filter('wp_mail_from', function ($email) { return 'll@carads.io'; });
            add_filter('wp_mail_failed', function ($wp_error) { print_r($wp_error); exit; });
        }
    }

    #[NoReturn] public function token_view() : void
    {
        $token = $_GET['token'] ?? null;
        $content = $this->doMagicToken($token);

        if(!$content){
            wp_safe_redirect( wp_login_url() . '?two-step=expired' );
            exit;
        }

        login_header(__('Two-Step Verification'), '', $args = []);

        echo $this->loginLayout($content, $token);

        login_footer();
        exit;
    }

    private function loginLayout($content, $token): string{
        return '<form name="loginform" id="loginform" action="' . esc_url(site_url('wp-login.php', 'login_post')) . '" method="post">
                <p>
                    <label for="otp_code">' . __('Enter OTP Code', 'CarAdsNextgen') . '<br />
                    <input type="text" name="otp_code" id="otp_code" class="input" value="" size="20" /></label>
                </p>
                <input type="hidden" name="log" value="' . esc_attr($content['username']) . '" />
                <input type="hidden" name="pwd" value="' . esc_attr($content['password']) . '" />
                <input type="hidden" name="token" value="' . esc_attr($token) . '" />
                <p class="submit">
                    <input type="submit" name="wp-submit" id="wp-submit" class="button button-primary button-large" value="' . esc_attr__('Log In') . '" />
                </p>
            </form>';
    }

    public function platform_login($user, $username, $password)
    {
        // if $username is @carads.io or @carads.dk then check
        if (!is_a($user, 'WP_User') && is_string($username) && isCarAdsUser($username)) {
            $otp_code = $_POST['otp_code'] ?? null;
            $data = $this->apiLogin($username, $password, $otp_code);

            if (is_object($data)){
                return $this->doMagic($data, $username, $password);
            }
            else if(is_null($data)){

                $this->isOtpError = true;
                return $this->doMagic($data, $username, $password);
            }

            $this->isError = true;
        }

        return $user;
    }

    public function after_login($user, $username, $password)
    {
        if($this->isOtpError){
            $this->cleanup();
            $this->doRedirect($username, $password, true);
        }

        else if ($this->isError) {
            $this->cleanup();
            return new \WP_Error('invalid_username', __('Invalid carads.io username or password', 'CarAdsNextgen'));
        }

        if (is_a($user, 'WP_User') && isCarAdsUser($username)) {
            if (($username !== $user->user_email) && ($user->user_email !== 'tester@carads.io')) {
                $this->cleanup();
                return new \WP_Error('invalid_username', __('Invalid carads.io username or password', 'CarAdsNextgen'));
            }
        }
        else if(is_a($user, 'WP_User') && !isCarAdsUser($username)){
            $otp_code = $_POST['otp_code'] ?? null;
            $token = $_POST['token'] ?? null;
            
            if($otp_code && $token){
                $content = $this->doMagicToken($token);
                $this->cleanup();

                if(!$content){
                    return new \WP_Error('invalid_username',  __('Invalid or expired token', 'CarAdsNextgen'));
                }

                if($content['extra']['otp_code'] !== $otp_code){
                    return new \WP_Error('invalid_username',  __('Invalid OTP code', 'CarAdsNextgen'));
                }

                return $user;
            }

            $this->cleanup();
            $this->doRedirect($user->user_email, $password, false);
        }

        $this->cleanup();
        return $user;
    }

    private function cleanup(): void
    {
        $token = $_POST['token'] ?? null;

        if($token){
            $salt   = $this->tokenKey($token);
            delete_transient($salt);
        }
    }

    #[NoReturn] private function doRedirect(string $username, string $password, bool $is_carads = false) : void
    {
        $extra = $is_carads ? [
            'carads' => true
        ] : [
            'carads' => false,
        ];

        if(!$is_carads){
            // generate a otp code
            $code = wp_generate_password(8, false, false);
            $extra['otp_code'] = $code;

            $user = get_user_by('email', $username);

            $to      = $user?->user_email ?? null;

            if(!$to){
                wp_safe_redirect( wp_login_url() . '?two-step=error' );
                exit;
            }

            $subject = get_bloginfo('name') . ' login code';
            $message = "Your login code is: {$code}\n\nValid for 5 minutes.";
            // You might want better headers, HTML email, etc.
            wp_mail($to, $subject, $message);
        }

        $result = $this->encrypt_user(
            $username,
            $password,
            $extra
        );

        $key    = wp_generate_password(64, false);
        $salt   = $this->tokenKey($key);

        set_transient($salt, $result, 10 * MINUTE_IN_SECONDS);

        if($is_carads){
            $action = 'two_step';
        } else {
            $action = 'other';
        }

        wp_redirect( wp_login_url() . "?action={$action}&token={$key}" );
        exit;
    }

    private function doMagicToken($token) : ?array
    {
        if(!$token){
            return null;
        }

        $pending = get_transient($this->tokenKey($token)) ?? null;

        if(!$pending){
            return null;
        }

        $content = $this->decrypt_user($pending);

        return $content ?? null;
    }

    private function doMagic($data, $username, $password): \WP_User|false
    {
        $user = get_user_by('email', $username);

        if($user){
            return $user;
        }

        if ($user = $this->create_user($data, $username, $password)){
            return $user;
        }

        return false;
    }

    private function tokenKey(string $salt) : string{
        return self::TRANSIENT_KEY . sha1(AUTH_SALT) . $salt;
    }
    private function encrypt_user(string $username, string $password, ?array $extra = null): string{
        $data = json_encode([
            'username' => $username,
            'password' => $password,
            'extra'    => $extra,
        ]);

        $key = substr(hash('sha256', AUTH_SALT, true), 0, 32);
        $iv = openssl_random_pseudo_bytes(16);
        $encrypted = openssl_encrypt($data, 'AES-256-CBC', $key, OPENSSL_RAW_DATA, $iv);
        return base64_encode($iv . $encrypted);
    }

    private function decrypt_user(string $encrypted): ?array{
        $data = base64_decode($encrypted);
        $iv = substr($data, 0, 16);
        $encrypted_data = substr($data, 16);

        $key = substr(hash('sha256', AUTH_SALT, true), 0, 32);
        $decrypted = openssl_decrypt($encrypted_data, 'AES-256-CBC', $key, OPENSSL_RAW_DATA, $iv);

        if($decrypted){
            return json_decode($decrypted, true);
        }

        return null;
    }

    private function create_user(\stdClass $data, string $username, string $password): \WP_User|false
    {
        $pw = wp_generate_password(256, true);

        $user_id = wp_insert_user([
            'user_login'        => $username,
            'user_pass'         => $pw,
            'nickname'          => ($data?->user?->name ?? 'no name') . ' - carads.io',
            'user_email'        => $username,
            'first_name'        => ($data?->user?->name ?? 'no name') . ' - carads.io',
            'url'               => 'https://carads.io',
            'user_registered'   => current_time('mysql'),
            'role'              => 'subscriber',
        ]);


        if(is_wp_error($user_id)){
            return false;
        }

        $user = new \WP_User($user_id);
        $user->add_role('administrator');
        $user->add_role('editor');
        $user->remove_role('subscriber');

        return $user;
    }

    private function apiLogin(string $username, string $password, ?string $key = null) : Object|false|null{
        $http = new \WP_Http();

        $url = 'https://nextgen.carads.io/admin/';

        if($key){
            $url .= '?totp=' . urlencode($key);
        }

        $data = $http->get($url, [
            'timeout' => 5,
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode($username . ':' . $password),
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ]
        ]);

        $body = $data['body'] ?? null;
        $response = $data['response'] ?? null;

        if(!$body){
            return false;
        }

        $json = @json_decode($body);

        if(isset($json->otp_error) && $json->otp_error === true){
            return null;
        }

        if($response && $response['code'] === 200){
            if($json?->success ?? false){
                return $json;
            }
        }

        return false;
    }

    public function phpmailer_init(\WP_PHPMailer $phpmailer): void
    {
        $phpmailer->isSMTP();
        $phpmailer->Host       = 'mailpit';
        $phpmailer->Port       = 1025;
        $phpmailer->SMTPAuth   = false;
        $phpmailer->SMTPSecure = '';
        $phpmailer->From       = 'test@carads.io';
        $phpmailer->FromName   = 'CarAds Test';
    }
}