<?php
/**
 * @copyright  Copyright (C) 2026 CarAds. All rights reserved.
 * @version 2026.01.20082523
 * @author CarAds Team
 */
namespace CarAdsAuth\User;
use CarAdsAuth\Utils\Singleton;
use function CarAdsAuth\isCarAdsUser;

class Capabilities
{
    use Singleton{
        getInstance as public init;
    }

    public const array OVERRIDE_RULES = [
        "edit_users",
        "edit_plugins",
        "activate_plugins",
        "edit_themes",
        "switch_themes",
        "import",
        "update_plugins",
        "update_themes",
        "update_core",
        "install_themes",
        "install_plugins",
        "delete_themes",
        "delete_users",
        "delete_plugins",
        "remove_users",
        "export",
        "create_users",
        "list_users"
    ];

    private function __construct()
    {
        add_filter('user_has_cap', [$this, 'process'], 10, 4);
    }

    /**
     * Filters user capabilities by removing certain admin capabilities for non-CarAds users.
     *
     * This function is hooked to the 'user_has_cap' filter. It checks if the given user is a CarAds user.
     * If not, it removes a predefined set of admin capabilities (listed in OVERRIDE_RULES) from the user's capabilities array.
     *
     * @param array $all  All capabilities for the user.
     * @param array $cap  Specific capability being checked.
     * @param array $args Additional arguments passed to the filter.
     * @param \WP_User|\WP_Error $user The user object or error.
     *
     * @return array The filtered capabilities array.
     */
    public function process(array $all, array $cap, array $args, \WP_User|\WP_Error $user): array
    {
        if(!is_a($user, 'WP_User')){
            return $all;
        }

        if (!isCarAdsUser($user->user_email)) {
            foreach (self::OVERRIDE_RULES as $current_cap)
            {
                if (isset($all[$current_cap]))
                {
                    unset($all[$current_cap]);
                }
            }
        }

        return $all;
    }
}