<?php
/**
 * @copyright  Copyright (C) 2026 CarShop (Part of CarAds). All rights reserved.
 * @version   2026.01.20150259
 * @author    CarAds Team
 */
namespace CarShop\Basic;

class Configs implements \Iterator , \ArrayAccess
{
    protected string $path;
    protected array $configFiles;
    protected array $cacheRawConfig = [];

    public function __construct(string $path){
        $this->path = $path;
        $this->configFiles = $this->correctPath($this->path);
    }

    /**
     * @param string $path
     *
     * @return array
     */
    private function correctPath(string $path) : array
    {
        if(!is_dir($path)){
            return [];
        }

        $dir = new \FilesystemIterator($path, \FilesystemIterator::SKIP_DOTS);

        if(!$dir->valid()){
            return [];
        }

        $files = [];
        foreach ($dir as $file) {
            if ($file->isFile() && $file->getExtension() === 'json') {
                $files[] = $file->getPathname();
            }
        }

        if(empty($files)){
            return [];
        }

        return array_filter(array_map(function($file) {
            return basename($file, '_config.json');
        }, $files), function($file) {
            return stripos($file, '_') !== 0;
        });
    }


    public function exists($name) : bool
    {
        return in_array($name, $this->configFiles);
    }

    public function current(): mixed
    {
        return $this->configFiles[$this->key()];
    }

    public function next(): void
    {
        next($this->configFiles);
    }

    public function key(): string|int|null
    {
        return key($this->configFiles);
    }

    public function valid(): bool
    {
        return key($this->configFiles) !== null;
    }

    public function rewind(): void
    {
        reset($this->configFiles);
    }

    public function offsetExists(mixed $offset): bool
    {
        return $this->exists($offset);
    }

    public function offsetGet(mixed $offset): null|string
    {
        if($this->exists($offset)){
            if(isset($this->cacheRawConfig[$offset])){
                return $this->cacheRawConfig[$offset];
            }

            $this->cacheRawConfig[$offset] = file_get_contents($this->path . '/' . $offset . '_config.json');

            return $this->cacheRawConfig[$offset];
        }

        return null;
    }

    public function offsetSet(mixed $offset, mixed $value): void
    {
    }

    public function offsetUnset(mixed $offset): void
    {
    }

    public function files() : array{
        return $this->configFiles;
    }

    public function path() : string{
        return rtrim($this->path, '/') . '/';
    }

    public function getRaw(string $name, $setDefault = null) : mixed
    {
        return $this->offsetGet($name) ?: $setDefault;
    }

    public function sha1(string $name) : string|null
    {
        $content = $this->getRaw($name);

        if($content !== null){
            return sha1($content);
        }

        return null;
    }

    public function filePath(string $name, $absolute = true) : string{
        if ($absolute){
            return $this->path() . $name . '_config.json';
        }

        return str_replace(ABSPATH, '', $this->path() . $name . '_config.json');
    }

    public function fileMTime(string $name) : int|null{
        $filePath = $this->filePath($name);
        if(file_exists($filePath)){
            return filemtime($filePath);
        }
        return null;
    }

    public function raw_files(): array
    {
        return array_map(function($item){
            return [
                'file' => $item,
                'sha1' => $this->sha1($item),
                'config' => json_decode($this->getRaw($item)),
                'time' => $this->fileMTime($item),
                'file_path' => $this->filePath($item, false),
            ];
        }, $this->configFiles);
    }
}