<?php
/**
 * @copyright  Copyright (C) 2026 CarShop (Part of CarAds). All rights reserved.
 * @version   2026.01.20150259
 * @author    CarAds Team
 */
namespace CarShop\Basic;
use stdClass;

/**
 * SEO management singleton class.
 *
 * @property string $title
 * @property string $description
 * @property string $canonical
 * @property string $robots
 * @property string $keywords
 * @property array $images
 */
class Seo
{
    protected bool $third_party_seo = false;

    protected bool $is_final = false;

    /**
     * @var string $title
     */
    protected string $title = '';
    /**
     * @var string $description
     */
    protected string $description = '';
    /**
     * @var string $canonical
     */
    protected string $canonical = '';
    /**
     * @var string $robots
     */
    protected string $robots = '';
    /**
     * @var string $keywords
     */
    protected string $keywords = '';
    /**
     * @var array $images
     */
    protected array $images = [];

    public function __set(string $name, $value): void
    {
        $method = 'set' . ucfirst($name);
        if (method_exists($this, $method)) {
            $this->$method($value);
            return;
        }

        throw new \InvalidArgumentException("Property {$name} does not exist.");
    }

    public function __get(string $name)
    {
        if (property_exists($this, $name)) {
            return $this->$name;
        }

        throw new \InvalidArgumentException("Property {$name} does not exist.");
    }

    protected function setTitle(string $title): void
    {
        $this->title = $title;
    }

    protected function setDescription(string $description): void
    {
        $this->description = str_replace("\n", " ", strip_tags($description));
    }

    protected function setCanonical(string $canonical): void
    {
        $this->canonical = $canonical;
    }

    protected function setRobots(string $robots): void
    {
        $this->robots = $robots;
    }

    protected function setKeywords(string $keywords): void
    {
        $this->keywords = str_replace("\n", " ", strip_tags($keywords));
    }

    protected function setImages(array $images): void
    {
        $this->images = $images;
    }

    protected function output(): stdClass
    {
        return (object)[
            'title'         => $this->title,
            'description'   => $this->description,
            'keywords'      => $this->keywords,
            'robots'        => $this->robots,
            'canonical'     => $this->canonical,
            'images'        => $this->images
        ];
    }

    public function __construct(array $settings)
    {
        $this->third_party_seo = $settings['third_party_seo'] ?? false;
        add_action('wp_head', [$this, 'inject'], 0, 1);

        if(!is_plugin_active('wordpress-seo-premium/wp-seo-premium.php')){
            add_action('wp_head', [$this, 'ob_start_callback'], -9999999, 1);
            add_action('wp_head', [$this, 'ob_end_callback'], 9999999, 1);
        }
    }

    public function _title($title) : array
    {
        $title['title'] = $this->title;

        if(isset($title['site'])){
            unset($title['site']);
        }

        return $title;
    }

    public function inject(): void
    {
        if(!is_plugin_active('wordpress-seo/wp-seo.php')){
            add_filter('document_title_parts', [$this, '_title'], 10, 1);
            return;
        }

        if($this->robots){
            add_filter('wpseo_robots', function() {
                return $this->robots;
            }, 0, 1);
        }

        if($this->canonical){
            add_filter('wpseo_canonical', function () { return $this->canonical; });
            add_filter('wpseo_opengraph_url', function () { return $this->canonical; });
        }

        if($this->title){
            add_filter('wpseo_title',     function () { return $this->title; });
            add_filter('wpseo_opengraph_title', function (){ return $this->title; });
            add_filter('wpseo_twitter_title', function (){ return $this->title; });
        }

        if($this->description){
            add_filter('wpseo_metadesc',  function () { return $this->description; });
            add_filter('wpseo_opengraph_desc', function (){  return $this->description; }); // not sure this is not supported for free when title is!!!
            add_filter('wpseo_twitter_description', function () { return $this->description; });
        }

        if(!is_plugin_active('wordpress-seo-premium/wp-seo-premium.php')){
            return;
        }

        add_filter('wpseo_add_opengraph_images', function ($images) {
            if(empty($this->images)){
                return;
            }

            foreach ($this->images as $image){
                $images->add_image_by_url($image);
            }
        });

        // https://developer.yoast.com/features/opengraph/api/wpseo-opengraph-images/
        add_filter('wpseo_opengraph_image', function($image){
            if(empty($this->images)){
                return $image;
            }

            return current($this->images);
        });

        add_filter('wpseo_twitter_image', function (){
            if(count($this->images) === 0){
                return null;
            }

            return current($this->images) ?? '';
        });

    }

    public function ob_start_callback(): void
    {
        ob_start();
    }

    public function ob_end_callback(): void
    {
        $html = ob_get_clean();

        if(!$html || $this->is_final){
            return;
        }

        list($newHTML, $code) = Head::output($html, $this->output(), $this->third_party_seo);

        if($this->third_party_seo && $code){
            $html = $newHTML;
        }

        $html = ($html . $code);
        $this->is_final = true;

        echo $html;
    }
}