<?php
/**
 * @copyright  Copyright (C) 2026 CarShop (Part of CarAds). All rights reserved.
 * @version   2026.01.20150259
 * @author    CarAds Team
 */
namespace CarShop\Basic;

use Exception;

/**
 * Class UpdateConfig
 *
 */
class UpdateConfig
{
    CONST int PLUGIN_TYPE_UNDEFINED = 0;
    CONST int PLUGIN_TYPE_CARADS = 1 << 1;
    CONST int PLUGIN_TYPE_CARSHOP = 1 << 2;
    CONST int PLUGIN_TYPE_DEALERSHIP = 1 << 3;
    CONST int PLUGIN_CARADS_MASTER = 1 << 1;
    CONST int PLUGIN_CARADS_SLOW = 1 << 2;
    CONST int PLUGIN_CARADS_SLIDER = 1 << 3;
    CONST int PLUGIN_CARADS_QUICK = 1 << 4;
    CONST int PLUGIN_CARSHOP_MASTER = 1 << 5;
    CONST int PLUGIN_CARSHOP_SLIDER = 1 << 6;
    CONST int PLUGIN_SITE_MAP = 1 << 7;
    CONST int PLUGIN_DEALERSHIP_DEALER = 1 << 8;
    const int PLUGIN_UNDEFINED = 0;
    const int OPERATION_SUCCESS = 0;
    const int ERROR_INVALID_COMPANY_ID = 1 << 0;
    const int ERROR_INVALID_RESPONSE = 1 << 1;
    const int ERROR_NO_CONFIGS_FOUND = 1 << 2;
    const int ERROR_CREATE_OR_UPDATE_FILE = 1 << 3;
    const int ERROR_DELETE_FILE = 1 << 4;
    const int DELETE_FILE_PLUGIN_UNDEFINED = 1 << 0;
    const int DELETE_FILE_PATH_UNDEFINED = 1 << 1;
    const int DELETE_FILE_PLUGIN_NOT_ACTIVE = 1 << 2;
    const int DELETE_FILE_NOT_EXISTS = 1 << 3;
    const int DELETE_FILE_NOT_WRITABLE = 1 << 4;
    const int CREATE_FILE_PLUGIN_UNDEFINED = 1 << 5;
    const int CREATE_FILE_PATH_UNDEFINED = 1 << 6;
    const int CREATE_FILE_PLUGIN_NOT_ACTIVE = 1 << 7;
    const int CREATE_FILE_CANNOT_MKDIR = 1 << 8;
    const int CREATE_FILE_JSON_ERROR = 1 << 9;
    const int CREATE_FILE_NOT_WRITABLE = 1 << 10;
    const int CREATE_FILE_CANNOT_WRITE_FILE = 1 << 11;


    static public function current_file_names():array{
        $configs = apply_filters('carads_config_files', []);
        $files = [];
        foreach ($configs as $config) {
            // if class has method files
            if (!method_exists($config, 'files')) {
                continue;
            }

            foreach ($config->files() as $file) {
                $info = self::get_types($file);

                if(static::PLUGIN_UNDEFINED === $info['type']){
                    continue;
                }

                $files[] = $file;
            }
        }

        return $files;
    }

    /**
     * @return array
     */
    static public function current_files(): array
    {
        $configs = apply_filters('carads_config_files', []);
        $files = [];

        foreach ($configs as $config) {
            // if class has method files
            if (!method_exists($config, 'raw_files')) {
                continue;
            }

            foreach ($config->raw_files() as $file) {
                $info = self::get_types($file['file']);

                if(static::PLUGIN_UNDEFINED === $info['type']){
                    continue;
                }

                $files[] = $file;
            }
        }

        return array_combine(array_column($files, 'file'), $files);
    }

    /**
     * Get path to plugin config folder
     * - sitemap can be in carads or carshop plugin folder - first found is returned - can be other cases later that need same logic
     *
     * @param int $plugin - it flag - so can be multiple plugins combined with - but it will return first found
     * @return string|null
     *
     */
    static private function get_path(int $plugin):?string{
        $path = WP_CONTENT_DIR . '/plugins/';

        if($plugin === self::PLUGIN_TYPE_UNDEFINED){
            return null;
        }

        if($plugin & static::PLUGIN_TYPE_CARADS && (is_plugin_active('c2/app.php') || is_plugin_active('carads/app.php'))){
            if(defined('CARSHOP_TEST_PATH') && CARSHOP_TEST_PATH){
                $path = CA_NEXTGEN_PATH . '/config/';
            } else {
                $path .= '.c2/';
            }

            return $path;
        }
        else if($plugin & static::PLUGIN_TYPE_CARSHOP && is_plugin_active('carshop/app.php')){
            if(defined('CARSHOP_TEST_PATH') && CARSHOP_TEST_PATH){
                $path = CARSHOP_NEXTGEN_PATH . '/config/';
            } else {
                $path .= '.carshop/';
            }

            return $path;
        }
        else if($plugin & static::PLUGIN_TYPE_DEALERSHIP && (is_plugin_active('dealership/app.php') || is_plugin_active('carads-dealership/app.php'))){
            if(defined('CA_DEALERSHIP_TEST_PATH') && CA_DEALERSHIP_TEST_PATH){
                $path = CA_DEALERSHIP_DIR . '/config/';
            } else {
                $path .= '.dealership/';
            }

            return $path;
        }

        return null;
    }

    /**
     * @throws Exception
     */
    static private function add_file(string $name, $content = '', bool $allow_update = false): bool
    {
        $info = self::get_types($name);

        if(static::PLUGIN_UNDEFINED === $info['type']){
            throw new Exception('Undefined plugin type', self::CREATE_FILE_PLUGIN_UNDEFINED);
        }

        if(!defined('WP_CONTENT_DIR')){
            throw new Exception('WP_CONTENT_DIR is not defined', self::CREATE_FILE_PATH_UNDEFINED);
        }

        $path = self::get_path($info['plugin']);

        if($path === null){
            throw new Exception('plugin is not defined', self::CREATE_FILE_PLUGIN_NOT_ACTIVE);
        }
        else if(!file_exists($path)){
            if(mkdir($path, 0755, true) === false){
                throw new Exception('Cannot create directory: ' . $path, self::CREATE_FILE_CANNOT_MKDIR);
            }
        }

        $full_path = $path . $name . '_config.json';
        $content = json_encode($content, JSON_PRETTY_PRINT);

        if(json_last_error() !== JSON_ERROR_NONE){
            throw new Exception('JSON encode error: ' . json_last_error_msg(), self::CREATE_FILE_JSON_ERROR);
        }

        if(file_exists($full_path) && !$allow_update){
            return false;
        }

        // check if folder is writable
        if(!is_writable($path)){
            throw new Exception('Directory is not writable: ' . $path, self::CREATE_FILE_NOT_WRITABLE);
        }
        else if(file_exists($full_path) && !is_writable($full_path)){
            throw new Exception('File is not writable: ' . $full_path, self::CREATE_FILE_CANNOT_WRITE_FILE);
        }

        return file_put_contents($full_path, $content) !== false;
    }

    /**
     * @return bool|null - if file does not exist return null - else return true if deleted or false if not deleted
     * throws exception on errors before last step
     *
     * @throws Exception
     */
    static private function delete_file(string $name): ?bool
    {
        $info = self::get_types($name);

        if(static::PLUGIN_TYPE_UNDEFINED === $info['plugin']){
            throw new Exception('Undefined plugin type', self::DELETE_FILE_PLUGIN_UNDEFINED);
        }

        if(!defined('WP_CONTENT_DIR')){
            throw new Exception('WP_CONTENT_DIR is not defined', self::DELETE_FILE_PATH_UNDEFINED);
        }

        $path = self::get_path($info['plugin']);

        if($path === null){
            throw new Exception('Plugin is not active', self::DELETE_FILE_PLUGIN_NOT_ACTIVE);
        }
        else if(!file_exists($path)){
            throw new Exception('Directory does not exist: ' . $path, self::DELETE_FILE_NOT_EXISTS);
        }
        else if(!is_writable($path)){
            throw new Exception('Directory is not writable: ' . $path, self::DELETE_FILE_NOT_WRITABLE);
        }

        $full_path = $path . $name . '_config.json';

        if(!file_exists($full_path)){
            return null;
        }

        return unlink($full_path);
    }

    /**
     * @param int $cid - company id to sync configs for.
     * @param bool $force - force update even if no configs found on carads server.
     * @return object{
     *     status: int,
     *     errors: array,
     *     success: array
     * }
     *
     */
    static function sync(int $cid, bool $force = false): object
    {
        if($cid === -1){
            // if company id is not set or is -1 then we don't need to update configs
            // because we don't have company id to sync configs
            return (object) [
                'status' => static::ERROR_INVALID_COMPANY_ID,
                'errors' => [],
                'success' => [],
            ];
        }

        $currentConfigFiles = self::current_files();
        $configs = self::fetch($cid);

        if(!is_array($configs)){
            return (object) [
                'status' => static::ERROR_INVALID_RESPONSE,
                'errors' => [],
                'success' => [],
            ];
        }
        else if(count($configs) === 0){
            return (object) [
                'status' => static::ERROR_INVALID_RESPONSE,
                'errors' => [],
                'success' => [],
            ];
        }

        $tmpLocal = array_combine(array_keys($currentConfigFiles), array_map(function($file){
            return $file . '_config.json';
        }, array_keys($currentConfigFiles)));
        $tmpLive = array_combine(array_keys($configs),array_map(function($file){
            return $file . '_config.json';
        }, array_keys($configs)));


        $news       = array_diff($tmpLive, $tmpLocal);
        $deleted    = array_keys(array_diff($tmpLocal, $tmpLive));
        $updates    = array_intersect($tmpLive, $tmpLocal);
        $do_files   = array_keys(array_merge($news, $updates));
        $errors     = [];
        $success    = [];

        if(count($tmpLive) === 0 && $force === false){
            return (object) [
                'status' => self::ERROR_NO_CONFIGS_FOUND,
                'errors' => [],
                'success' => [],
            ];
        }

        $success_create_or_update = true;
        foreach($do_files as $file){
            try{
                $res = self::add_file($file, 'tester', true);
                if($res === false){
                    $errors[$file] = 'cannot.add.file';
                    $success_create_or_update = false;
                } else {
                    $success[] = $file;
                }
            }
            catch(Exception $e){
                $errors['sync::' . str_replace('.', '_', $file)] = [
                    'message' => $e->getMessage(),
                    'code' => $e->getCode(),
                    'file' => $e->getFile() . ':' . $e->getLine()
                ];
                $success_create_or_update = false;
            }
        }

        $success_delete = true;
        foreach($deleted as $file){
            try{
                $res = self::delete_file($file);
                if($res === false){
                    $errors[$file] = 'cannot.delete.file';
                    $success_delete = false;
                } else if($res === null){
                    $errors[$file] = 'file.not.exists';
                    $success_delete = false;
                } else {
                    $success[] = $file;
                }
            }
            catch(Exception $e){
                $success_delete = false;
                $errors['delete::' . str_replace('.', '_', $file)] = [
                    'message' => $e->getMessage(),
                    'code' => $e->getCode(),
                    'file' => $e->getFile() . ':' . $e->getLine()
                ];
            }
        }

        if(!$success_create_or_update || !$success_delete){
            $status = 0;
            if(!$success_create_or_update){
                $status |= static::ERROR_CREATE_OR_UPDATE_FILE;
            }
            if(!$success_delete){
                $status |= static::ERROR_DELETE_FILE;
            }

            return (object) [
                'status' => $status,
                'errors' => $errors,
                'success' => $success,
            ];
        }

        return (object) [
            'status' => static::OPERATION_SUCCESS,
            'errors' => $errors,
            'success' => $success,
        ];
    }


    /**
     *
     * @param int $status
     * @param bool $as_string
     *
     * - $as_string return as string or array - static::ERROR_CREATE_OR_UPDATE_FILE and static::ERROR_DELETE_FILE
     * | can have multiple errors combined. but old system only one error message is returned.
     * | so for not breaking changes - if $as_string is true - only first error message is returned.
     *
     * @return string|array
     */
    static function errorCodeMessages(int $status, bool $as_string = true) : string|array{
        $message = [];
        if($status === static::OPERATION_SUCCESS) {
            $message[] = 'configs.updated';
        }
        else if($status & static::ERROR_INVALID_COMPANY_ID) {
            $message[] = 'invalid.company.id';
        }
        else if($status & static::ERROR_INVALID_RESPONSE) {
            $message[] = 'invalid.response';
        }
        else if($status & static::ERROR_NO_CONFIGS_FOUND) {
            $message[] = 'no.configs.found';
        }

        if($status & static::ERROR_CREATE_OR_UPDATE_FILE) {
            $message[] = 'configs.update.failed';
        }
        if($status & static::ERROR_DELETE_FILE) {
            $message[] = 'configs.cleanup.failed';
        }

        if(count($message) === 0){
            $message = ['unknown.error'];
        }

        if($as_string){
            return current($message);
        }

        return $message;
    }
    /**
     */
    static function isSuccessStatus($status): bool{
        return $status === static::OPERATION_SUCCESS;
    }

    /**
     */
    static function errorMessages($status, $separator = ' & ', bool $only_one = false) : string{
        $message = [];
        if($status === static::OPERATION_SUCCESS) {
            $message[] = 'Operation success';
        }
        else if($status & static::ERROR_INVALID_COMPANY_ID) {
            $message[] = 'Error: Invalid company ID';
        }
        else if($status & static::ERROR_INVALID_RESPONSE) {
            $message[] = 'Error: Invalid response from server';
        }
        else if($status & static::ERROR_NO_CONFIGS_FOUND) {
            $message[] = 'Error: No configs found';
        }

        if($status & static::ERROR_CREATE_OR_UPDATE_FILE) {
            $message[] = 'Error: Cannot create or update file';
        }
        if($status & static::ERROR_DELETE_FILE) {
            $message[] = 'Error: Cannot delete file';
        }

        if(count($message) === 0){
            return 'Der opstod en ukendt fejl.';
        }
        else if($only_one){
            return current($message);
        }

        return join($separator, $message);
    }

    /**
     * @param string $input
     *
     * @return array
     *
     */
    static function get_types(string $input): array
    {
        if(str_ends_with($input, '_catalog')){
            return static::get_types_catalog($input);
        }
        else if(str_contains($input, 'dealer')){
            return self::get_types_dealership($input);
        }
        else if($input === 'sitemap'){
            return self::get_types_sitemap($input);
        }

        return static::get_types_carads($input);
    }

    /**
     * @param string $input
     * @return array
     *
     */
    private static function get_types_carads(string $input): array
    {
        $parts  = explode('_', $input);
        $type   = end($parts);
        array_pop($parts);
        $name   = implode('_', $parts);

        $types = [
            'master' => static::PLUGIN_CARADS_MASTER,
            'slow'   => static::PLUGIN_CARADS_SLOW,
            'slider' => static::PLUGIN_CARADS_SLIDER,
            'quick'  => static::PLUGIN_CARADS_QUICK,
        ];

        return [
            'name'      => $name,
            'type'      => $types[$type] ?? static::PLUGIN_UNDEFINED,
            'plugin'    => isset($types[$type]) ? static::PLUGIN_TYPE_CARADS : static::PLUGIN_TYPE_UNDEFINED,
            'raw_type'  => $type,
        ];
    }

    /**
     * @param string $input
     * @return array
     *
     */
    private static function get_types_sitemap(string $input): array
    {
        return [
            'name' => $input,
            'type' => static::PLUGIN_SITE_MAP,
            'plugin' => static::PLUGIN_TYPE_CARADS|static::PLUGIN_TYPE_CARSHOP,
        ];
    }

    /**
     * @param string $input
     * @return array
     *
     */
    private static function get_types_dealership(string $input): array
    {
        return [
            'name' => $input,
            'type' => static::PLUGIN_DEALERSHIP_DEALER,
            'plugin' => static::PLUGIN_TYPE_DEALERSHIP,
        ];
    }

    /**
     * @param string $input
     * @return array
     *
     */
    private static function get_types_catalog(string $input): array
    {
        $input = substr($input, 0, -8);

        if(str_ends_with($input, '_slider')){
            $input = substr($input, 0, -7);
            $type = static::PLUGIN_CARSHOP_SLIDER;
        } else {
            $type = static::PLUGIN_CARSHOP_MASTER;
        }

        return ['name' => $input, 'type' => $type, 'plugin' => static::PLUGIN_TYPE_CARSHOP];
    }

    /**
     * @param int $type
     * @return string
     *
     */
    static function types_name(int $type): string
    {
        return match ($type) {
            static::PLUGIN_CARADS_MASTER => 'master',
            static::PLUGIN_CARADS_SLOW   => 'slow',
            static::PLUGIN_CARADS_SLIDER => 'slider',
            static::PLUGIN_CARADS_QUICK  => 'quick',
            static::PLUGIN_CARSHOP_MASTER => 'catalog',
            static::PLUGIN_CARSHOP_SLIDER => 'slider_catalog',
            static::PLUGIN_SITE_MAP       => 'sitemap',
            static::PLUGIN_DEALERSHIP_DEALER => 'dealer',
            default => 'undefined',
        };
    }

    /**
     * @param string|int $cid
     * @return array|null
     *
     */
    private static function fetch(string|int $cid): ?array
    {
        $url = "https://nextgen.carads.io/virtuelconfig/list/{$cid}";

        $data = (new \WP_Http())->get($url);
        $response = $data['response'] ?? null;

        if ($response && $response['code'] === 200) {
            $body = json_decode($data['body'] ?? []);
            $ar = $body->configs ?? [];

            return array_combine(array_map(function($item){
                if($item->type === 'sitemap'){
                    return $item->slug;
                }

                return $item->slug . '_' . $item->type;
            }, $ar), $ar);
        }

        return null;
    }

    /**
     * @param int $id
     * @param bool $force
     * @return void
     *
     */
    public static function notices(int $id, bool $force = false): void{
        $response = self::sync($id, $force);
        $status = $response->status ?? 0;
        $errors = $response->errors ?? [];
        $errorMessage = self::errorMessages($status);

        if($status === static::OPERATION_SUCCESS) {
            add_action('admin_notices', function () use ($response) {
                echo '<div class="notice notice-success is-dismissible">';
                echo '<p>Configs updated successfully.</p>';
                echo '</div>';
            });
        } else if($status === self::ERROR_NO_CONFIGS_FOUND){
            add_action('admin_notices', function() use($errorMessage){
                $token = $_GET['token'] ?? '';
                echo '<div class="notice notice-warning is-dismissible">';
                echo '<p>' . $errorMessage . '</p>';
                echo '<a href="' . admin_url('?token=' . $token . '&tasks=sync&force=1') . '" class="button button-primary">Force sync configs</a>';
                echo '</div>';
            });
        } else {
            add_action('admin_notices', function() use($response, $errorMessage, $errors){
                echo '<div class="notice notice-error is-dismissible">';
                echo '<p>' . $errorMessage . '</p>';
                echo '<ul>';
                foreach($errors as $file => $error){
                    echo '<li>' . $file . ': ' . (is_array($error) ? ($error['message'] ?? 'unknown error') : $error) . '</li>';
                }
                echo '</ul>';
                echo '</div>';
            });
        }
    }
}