<?php
/**
 * @copyright  Copyright (C) 2026 Shared. All rights reserved.
 */

namespace CarShop\Shortcode;
use CarShop\Basic\ElementorWidget;
use CarShop\Basic\Shortcode;
use stdClass;
use function CarShop\Basic\createAttributes;
use function CarShop\Basic\current_factory;
use function CarShop\Basic\fetch_pre_rendered;

/**
 * Class Contact
 *
 * @example
 * [car-ads-contact cid="2301" slug="carads-webanalyse" layout="floating-outlined"][/car-ads-contact]
 * @example
 * [car-ads-contact]
 * {"cid":"2301", "slug":"carads-webanalyse", "layout":"floating-outlined"}
 * [/car-ads-contact]
 */
class Contact extends Shortcode
{
	/**
	 * @var string
	 */
	protected string $tag = 'car-ads-contact';

    protected string $class = 'carads-f';

	public function render($args = [], $content = null) : string
	{
        if($content !== null){
            $config = (object) @json_decode(trim($content));
        } else {
            $config = new stdClass;
        }
        $cid        = $args['cid'] ?? $args['id'] ?? $config?->cid ?? (isset(current_factory()->settings->cid) ? current_factory()->settings->cid : -1);
        $slug       = $args['slug'] ?? $config?->slug ?? '';
        $layout     = $args['layout'] ?? $config?->layout ?? 'floating-outlined';
        $pre_render = $args['pre_render'] ?? $config?->pre_render ?? true;
        $api        = match ($this->class){
            'service-f' => 'service-ads',
            default => 'nextgen',
        };

        $config->cid    = $cid;
        $config->slug   = $slug;
        $config->layout = $layout;

        $configString = json_encode($config);
        $body = '';

        // only render simple forms
        if($pre_render && count(get_object_vars($config)) <= 3){
            $body = $this->fetch($cid, $slug, $layout, $api);
        }

        return $this->innerHtml($configString, $body, $this->class);
	}

    private function innerHtml(string $config, string $body, string $class): string
    {
        $content = empty($body) ? 'Henter form' : $body;

        return <<<HTML
        <div class="{$class} carads" data-settings='{$config}'>
            {$content}
        </div>
HTML;
    }

    protected function fetch(int $cid, string $slug, string $layout, string $api): string
    {
        if(defined('WP_CONTENT_DIR')){
            $cacheDir = WP_CONTENT_DIR . '/cache/carads-nextgen/contact-forms/';
            if(!file_exists($cacheDir)){
                mkdir($cacheDir, 0755, true);
            }

            $cacheFile = $cacheDir . "form-{$cid}-{$slug}-{$layout}-{$api}2.html";

            if(file_exists($cacheFile) && (time() - filemtime($cacheFile) < 43200)){
                return file_get_contents($cacheFile);
            }

            $content = fetch_pre_rendered("https://ssr.js.nextgen.carads.io/form/{$cid}/{$slug}/{$layout}?api={$api}");

            $content = '<!-- Cached copy, generated at ' . date('Y-m-d H:i:s') . ' -->' . PHP_EOL . $content;

            file_put_contents($cacheFile, $content);

            return $content;
        }
    }

    public function elementor_register() : void{
        add_action('elementor/widgets/register', function($widgets_manager) {
            $class = new class extends ElementorWidget {
                protected string $title = 'CarAds Contact Form';
                protected string $tag = 'CarAdsContactForm';
                protected string $icon = 'eicon-mail';
                protected array $category = ['carads-widgets'];

                protected function register_controls(): void
                {
                    $this->start_controls_section(
                        'section_content',
                        [
                            'label' => __('Content', 'Shared'),
                        ]
                    );
                    $this->add_control(
                        'cid',
                        [
                            'label' => __('Contact Form CID', 'Shared'),
                            'type' => \Elementor\Controls_Manager::TEXT,
                            'default' => '',
                        ]
                    );
                    $this->add_control(
                        'slug',
                        [
                            'label' => __('Contact Form Slug', 'Shared'),
                            'type' => \Elementor\Controls_Manager::TEXT,
                            'default' => '',
                        ]
                    );
                    $this->add_control(
                        'layout',
                        [
                            'label' => __('Layout', 'Shared'),
                            'type' => \Elementor\Controls_Manager::SELECT,
                            'options' => [
                                'floating-outlined' => __('Floating Outlined', 'Shared'),
                                'boxed-filled' => __('Boxed Filled', 'Shared'),
                                'simple' => __('Simple', 'Shared'),
                            ],
                            'default' => 'floating-outlined',
                        ]
                    );
                    $this->add_control(
                        'pre_render',
                        [
                            'label' => __('Use Pre-rendered Content', 'Shared'),
                            'type' => \Elementor\Controls_Manager::SWITCHER,
                            'label_on' => __('Yes', 'Shared'),
                            'label_off' => __('No', 'Shared'),
                            'return_value' => 'yes',
                            'default' => 'yes',
                        ]
                    );
                    $this->end_controls_section();
                }

                protected function render(): void
                {
                    $s = $this->get_settings_for_display();
                    $attr = [
                        'cid' => $s['cid'] ?? '',
                        'slug' => $s['slug'] ?? '',
                        'layout' => $s['layout'] ?? 'floating-outlined',
                        'pre_render' => $s['pre_render'] === 'yes' ? 1 : 0,
                    ];

                    $attr = array_filter($attr, function($value){
                        return !empty($value);
                    });

                    $attr = createAttributes($attr);

                    echo do_shortcode('[car-ads-contact ' . $attr . '][/car-ads-contact]');
                }
            };

            if($this instanceof ServiceContact){
                $class = $class->createCustomInstance(
                    'ServiceAds Contact Form',
                    'ServiceAdsContactForm',
                    'eicon-mail',
                    ['carads-widgets']
                );
            }

            $widgets_manager->register($class);
        });
    }
}