<?php
/**
 * @copyright  Copyright (C) 2026 CarSHOP (Part of CarAds) All rights reserved.
 * @version 2026.01.20150259
 * @author CarAds Team
 */
namespace CarShop\Utils;
use CarAdsNextgen\Factory as C2Factory;
use CarShop\Basic\BasicSettings;
use Exception;
use function CarShop\factory;

/**
 * @property-read int $cid
 * @property-read string $script
 * @property-read string $api_key
 * @property-read string $secure_key
 * @property-read bool $carlead_enabled
 * @property-read bool $optimize
 * @property-read bool $shadow_root
 * @property-read bool $third_party_seo
 * @property-read string $form_layout
 */
class Settings extends BasicSettings {
    protected array $settings = [
        'cid'                   => ['CarShopClientID', -1, 'int', 1],
        'script'                => ['CarShopScript', 'master', 'string', 1],
        'api_key'               => ['CarShopApiKey', '', 'string', 1],
        'secure_key'            => ['CarShopSecureKey', '', 'string', 1],
        'carlead_enabled'       => ['CarShopCarleadEnabled', false, 'bool', 1],
        'optimize'              => ['CarShopOptimizeAssets', false, 'bool', 0],
        'shadow_root'           => ['CarShopShadowRoot', false, 'bool', 0],
        'third_party_seo'       => ['CarShopThirdPartySeo', false, 'bool', 0],
        'form_layout'           => ['CarShopFormLayout', 'standard', 'string', 0],
    ];

    /**
     * Proxy read access to CarAds Nextgen (C2) settings when that integration is present.
     *
     * When one of the listed settings is requested we call into the external C2 settings
     * implementation so any side-effects, migrations or cache updates in that implementation
     * can run. This keeps the two systems in sync when the C2 integration exists.
     *
     * Important behaviour and rationale:
     * - The call to `C2Factory::getInstance()->settings->__get($name)` is performed for its
     *   side-effects only. The return value is intentionally ignored and the value from
     *   `parent::__get($name)` is returned. This preserves local `BasicSettings` as the
     *   authoritative source and avoids surprising changes in behaviour for callers.
     * - The call is guarded with `factory()->c2Exists` so the code is safe when C2 is not
     *   installed or available.
     * - This design avoids potential recursion or precedence conflicts if the external
     *   implementation attempts to read back from this settings object.
     *
     * @param string $name
     * @return mixed
     * @throws Exception
     */
    public function __get($name){
        switch ($name){
            case 'cid':
            case 'script':
            case 'api_key':
            case 'secure_key':
            case 'carlead_enabled':
            case 'optimize':
            case 'shadow_root':
            case 'form_layout':
                if(factory()->c2Exists){
                    C2Factory::getInstance()->settings->__get($name);
                }

                return parent::__get($name);
            default:
                return parent::__get($name);
        }
    }

    /**
     * Proxy write access to CarAds Nextgen (C2) settings when that integration is present.
     *
     * When one of the listed settings is updated we call into the external C2 settings
     * implementation so any side-effects, migrations or cache updates in that implementation
     * can run. This keeps the two systems in sync when the C2 integration exists.
     *
     * @param string $name
     * @param mixed $value
     * @param mixed $autoload
     * @param bool $force
     * @return void
     * @throws Exception
     */
    public function update(string $name, mixed $value, $autoload = null, $force = true): void
    {
        switch ($name){
            case 'cid':
            case 'script':
            case 'api_key':
            case 'secure_key':
            case 'carlead_enabled':
            case 'optimize':
            case 'shadow_root':
            case 'form_layout':
                if(factory()->c2Exists){
                    C2Factory::getInstance()->settings->update($name, $value, $autoload, $force);
                }
                parent::update($name, $value, $autoload, $force);
                break;
            default:
                parent::update($name, $value, $autoload, $force);
                break;
        }
    }
}