<?php
/**
 * @copyright  Copyright (C) 2026 CarSHOP (Part of CarAds) All rights reserved.
 * @version 2026.01.20150259
 * @author CarAds Team
 */
namespace CarShop\Wordpress;

use CarShop\Basic\Seo;
use CarShop\Basic\Singleton;
use CarShop\Controller\CatalogController as CarshopCatalogController;
use CarShop\Controller\CatalogProductController as CarshopCatalogProductController;
use function CarShop\Basic\css_inject_helper;
use function CarShop\Basic\elementor_reload_preview;
use function CarShop\Basic\simple_cdn_version;
use function CarShop\factory;

class Rewrite
{
    use Singleton;
    /**
     * Maps route identifiers to their corresponding controller classes.
     * This constant array allows dynamic resolution of controllers based on the active route,
     * simplifying route management and improving code maintainability.
     */
    const array CONTROLLER = [
        'catalog-list'      => CarshopCatalogController::class,
        'catalog-product'   => CarshopCatalogProductController::class,
    ];

    private function __construct()
    {
        $route = factory()->route; // Get the singleton Route instance.


        add_filter('wp_head', function(){
            echo '<link rel="dns-prefetch" href="https://nextgen.carads.io" />';
            echo '<link rel="dns-prefetch" href="https://js.nextgen.carads.io" />';
            echo '<link rel="preconnect" href="https://nextgen.carads.io" />';
            echo '<link rel="preconnect" href="https://js.nextgen.carads.io" />';
        }, 11, 1);

        add_filter('wp_enqueue_scripts', [$this, 'css'], 99);
        add_filter('wp_script_attributes', function (array $data) {
            if($data['id'] === 'carads-main-js-module'){
                // https://developer.mozilla.org/en-US/docs/Web/HTML/Reference/Attributes/crossorigin
                $data['crossorigin'] = 'anonymous';
            }

            return $data;
        }, 10, 2);

        add_action('init', function () use($route) {
            $route->register_rewrites();   // Register custom rewrite rules.
            // If the route is active, trigger rendering logic for the matched route.
            if ($route->active()) {
                $this->render();
            }
        });

        add_filter('elementor/preview/enqueue_scripts', function(){
            elementor_reload_preview();
        }, 999);
    }

    public function css() : void{
        $file = $this->cssFile();

        wp_enqueue_style('carads-main', $file, [], null);

        $settings = factory()->settings;

        if ($settings->optimize){
            wp_enqueue_style('car-ads-icons', css_inject_helper('icons.css'), [], null);
            wp_enqueue_style('car-ads-opening-times', css_inject_helper('department/opening_times/basic.css'));

            /**
             * Before aksing why id is not same.
             * - virtual config settings cache will not reload php - so for javascript can use correct css,
             * id is make this way to make sure it is unique and we only need to add not update css file by id
             */
            if ($settings->form_layout === 'floating'){
                wp_enqueue_style('car-ads-form-floating', css_inject_helper('forms/floating-form.css'), [], null);
            }
            else if($settings->form_layout === 'floating-outlined'){
                wp_enqueue_style('car-ads-form-floating-outlined', css_inject_helper('forms/floating-outlined-form.css'), [], null);
            }
            else if($settings->form_layout === 'old-school'){
                wp_enqueue_style('car-ads-form-old', css_inject_helper('forms/old-school.css'), [], null);
            }
            else {
                wp_enqueue_style('car-ads-form-all', css_inject_helper('forms/_form.css'), [], null);
            }
        }

        $src = $this->jsFile();

        wp_register_script_module(
            'carads-main',
            $src,
            []
        );


        wp_enqueue_script_module('carads-main');
        if ($settings->carlead_enabled) {
            wp_enqueue_script('car-ads-carlead', 'https://nextgen.carads.io/assets/js/carlead.min.js?v=' . time() . '&wx=' . CA_NEXTGEN_VERSION);
        }
    }

    public function cssFile() : ?string{
        return css_inject_helper(factory()->settings->optimize ? 'optimize.css' : 'style.css');
    }

    /**
     * Returns the URL of the main JavaScript file for the frontend.
     *
     * Determines the correct source based on the environment:
     * - If `CA_TEST` is true or the CDN version is `localhost` or `local`, returns the local development URL.
     * - Otherwise, returns the production CDN URL.
     *
     * The query string is built from `mainJsQuery()`.
     *
     * @return string The full URL to the main JavaScript file.
     */
    public function jsFile() : string{
        $version = simple_cdn_version();
        $query = http_build_query($this->mainJsQuery());

        return (CARSHOP_TEST || ($version === 'localhost' || $version === 'local')) ?
            ("http://localhost:3000/src/main.js?{$query}") :
            ("https://js.nextgen.carads.io/{$version}/assets/main.js?{$query}");
    }

    /**
     * Builds the query parameters for the main JavaScript file.
     *
     * @return array
     */
    private function mainJsQuery() : array {
        $settings = factory()->settings;

        return [
            'v' => CARSHOP_TIME,
            'wx' => CARSHOP_NEXTGEN_VERSION,
            'environment' => $settings->cid,
            'shadow_root' => $settings->shadow_root ? 1 : 0,
        ];
    }

    private function render(): void
    {
        $active = factory()->route->active();
        $this->optimizeTemplateRedirect();

        if(!isset(self::CONTROLLER[$active->carshop])){
            return;
        }

        $seo = new Seo([
            '3rd_party_seo_active' => factory()->settings->third_party_seo,
        ]);

        add_action('template_redirect', function() use($active, $seo){
            $controller = self::CONTROLLER[$active->carshop];
            (new $controller($seo, factory()->route))->render();
        }, 99);
    }

    /**
     * Removes default WordPress template redirect actions that may interfere with custom routing.
     * If the `c2Exists` flag is set in the factory, the function exits early.
     * Otherwise, it hooks into `template_redirect` at priority 9 and unsets
     * `wp_old_slug_redirect` and `redirect_canonical` from the filter stack.
     */
    protected function optimizeTemplateRedirect() : void
    {
        /** @noinspection DuplicatedCode */
        add_action('template_redirect', function(){
            /**  */
            global $wp_filter, $wp_filters;

            $tmp = $wp_filter['template_redirect'][10];
            foreach($tmp AS $key => $value){
                if($value['function'] === 'wp_old_slug_redirect'){
                    unset($tmp[$key]);
                }
                if($value['function'] === 'redirect_canonical'){
                    unset($tmp[$key]);
                }
            }

            $wp_filter['template_redirect'][10] = $tmp;
        }, 9);
    }
}