<?php
/**
 * @copyright  Copyright (C) 2026 . All rights reserved.
 * @version   2026.01.20082522
 * @author    CarAds Team
 */
namespace CarAdsDealership\Basic;
use stdClass;

/**
 * Class Head
 *
 */
class Head
{
    const array DETECT = [
        'title'                 => 'title',
        'twitter:title'         => 'title',
        'og:title'              => 'title',
        'description'           => 'description',
        'twitter:description'   => 'description',
        'og:description'        => 'description',
        'twitter:image'         => 'images',
        'og:image'              => 'images',
        'canonical'             => 'canonical',
        'og:url'                => 'canonical',
        'robots'                => 'robots',
    ];

    /**
     * @param string $html
     * @param stdClass $ob
     * @param bool $remove
     * @return array This method will add meta tags to the head of the html page - in some cases seo yoast plugin will not added if you
     *
     * This method will add meta tags to the head of the html page - in some cases seo yoast plugin will not added if you
     * use free version - so we have added own support for it for page we control - like product page
     *
     * before method have be part of seo but for fast to called without all other event inside seo, we have move to own class
     * and make it static so we can call it directly without need to create object
     *
     *
     */
    static public function output(string $html, stdClass $ob, bool $remove = false): array
    {
        $newHeadContent = [];
        $cleanUp = false;
        $find = self::values($ob);

        /**
         * For ignore rules for some special tags - like twitter:image, og:image, og:url, canonical
         */
        $findSpecial = [
            'twitter:image', 'og:image',
            'og:url', 'canonical'
        ];

        foreach ($find as $element => $value){
            $isRemove = in_array($element, $findSpecial) || $remove;

            if(empty($value)){
                if($isRemove && self::isElementExist($element, $html)){
                    $html = self::removeElement($element, $html);
                    $cleanUp = true;
                }
                continue;
            } else if(is_array($value) && count($value) === 0){
                if($isRemove && self::isElementExist($element, $html)){
                    $html = self::removeElement($element, $html);
                    $cleanUp = true;
                }
                continue;
            }

            if(!self::isElementExist($element, $html)){
                $newHeadContent[] = self::addElement($element, $ob);
            }
            else if($isRemove) {
                $html = self::removeElement($element, $html);
                $cleanUp = true;
                $newHeadContent[] = self::addElement($element, $ob);
            }
        }

        if($cleanUp){
            // remove all empty lines
            $html = preg_replace('/\n\s*\n/', "\n", $html);
        }

        $code = "<!-- Shared SEO -->\n";
        $code .= implode("\n", $newHeadContent) . "\n";
        $code .= "<!-- /Shared SEO -->\n";

        return [$html, $code];
    }

    /**
     * @param string $element
     * @param string $html
     * @return bool
     *
     */
    static private function isElementExist(string $element, string $html) : bool{
        return match($element){
            'title' => str_contains($html, '<title>'),
            'description' => str_contains($html, '<meta name="description"'),
            'twitter:title' => str_contains($html, '<meta name="twitter:title"'),
            'twitter:description' => str_contains($html, '<meta name="twitter:description"'),
            'twitter:image' => str_contains($html, '<meta name="twitter:image"'),
            'og:title' => str_contains($html, '<meta property="og:title"'),
            'og:description' => str_contains($html, '<meta property="og:description"'),
            'og:image' => str_contains($html, '<meta property="og:image"'),
            'canonical' => str_contains($html, '<link rel="canonical"'),
            'og:url' => str_contains($html, '<meta property="og:url"'),
            'robots' => str_contains($html, '<meta name="robots"'),
            default => false
        };
    }

    /**
     * @param string $element
     * @param object $ob
     *
     * @return string|null
     *
     */
    static private function addElement(string $element, object $ob): ?string{
        return match($element){
            'title' => "<title>{$ob->title}</title>",
            'description' => "<meta name=\"description\" content=\"" . htmlentities($ob->description, ENT_QUOTES, 'UTF-8') . "\" />",
            'twitter:title' => "<meta name=\"twitter:title\" content=\"" . htmlentities($ob->title, ENT_QUOTES, 'UTF-8') . "\" />",
            'twitter:description' => "<meta name=\"twitter:description\" content=\"" . htmlentities($ob->description, ENT_QUOTES, 'UTF-8') . "\" />",
            'twitter:image' => implode("\n", array_map(function($image){
                return "<meta name=\"twitter:image\" content=\"{$image}\" />";
            }, $ob->images)),
            'og:title' => "<meta property=\"og:title\" content=\"" . htmlentities($ob->title, ENT_QUOTES, 'UTF-8') . "\" />",
            'og:description' => "<meta property=\"og:description\" content=\"". htmlentities($ob->description, ENT_QUOTES, 'UTF-8') ."\" />",
            'og:image' => '<meta property="og:image" content="' . current($ob->images) . '" />',
            'canonical' => "<link rel=\"canonical\" href=\"{$ob->canonical}\" />",
            'og:url' => "<meta property=\"og:url\" content=\"{$ob->canonical}\" />",
            'robots' => "<meta name=\"robots\" content=\"{$ob->robots}\" />",
            default => null
        };
    }

    /**
     * @param string $element
     * @param string $html
     * @return string
     *
     */
    static private function removeElement(string $element, string $html) : string{
        return match($element){
            'title' => preg_replace('/<title>.*<\/title>/', '', $html),
            'description' => preg_replace('/<meta name="description".*?>/', '', $html),
            'twitter:title' => preg_replace('/<meta name="twitter:title".*?>/', '', $html),
            'twitter:description' => preg_replace('/<meta name="twitter:description".*?>/', '', $html),
            'twitter:image' => preg_replace('/<meta name="twitter:image".*?>/', '', $html),
            'og:title' => preg_replace('/<meta property="og:title".*?>/', '', $html),
            'og:description' => preg_replace('/<meta property="og:description".*?>/', '', $html),
            'og:image' => preg_replace('/<meta property="og:image".*?>/', '', $html),
            'canonical' => preg_replace('/<link rel="canonical".*?>/', '', $html),
            'og:url' => preg_replace('/<meta property="og:url".*?>/', '', $html),
            'robots' => preg_replace('/<meta name="robots".*?>/', '', $html),
            default => $html
        };
    }

    static private function values(stdClass $ob) : array
    {
        $tmp = [];
        foreach (self::DETECT as $element => $key){
            if(property_exists($ob, $key)){
                $tmp[$element] = $ob->$key;
            }
            else {
                $tmp[$element] = null;
            }
        }

        return (array) $tmp;
    }
}