<?php
/**
 * @copyright  Copyright (C) 2026 . All rights reserved.
 * @version   2026.01.20082522
 * @author    CarAds Team
 */
namespace CarAdsDealership\Basic;
use DOMDocument;
use DOMXPath;
use Exception;
use Random\RandomException;
use stdClass;
use Elementor\Plugin as ElementorPlugin;
/**
 * Renders a PHP file with provided data.
 *
 * Extracts variables from `$data`, starts output buffering,
 * includes the specified file, and returns the buffered output as a string.
 *
 * @param string $file Path to the PHP file to include.
 * @param array|stdClass $data Data to extract as variables for use in the included file.
 * @return string The rendered output as a string.
 *
 */
function render(string $file, array|stdClass $data) : string
{
    extract((array) $data);

    ob_start();
    include func_get_arg(0);

    return ob_get_clean();
}

/**
 * @param int $r
 * @param int $g
 * @param int $b
 * @return string
 *
 */
function rgb(int $r, int $g, int $b) : string
{
    // return color in hex
    return sprintf("#%02x%02x%02x", $r, $g, $b);
}

/**
 * Replaces or adds a WordPress option.
 *
 * If the option does not exist, it adds it with the specified value.
 * If the option exists, it updates it with the new value.
 *
 * @param string $option The name of the option to replace or add.
 * @param mixed $value The value to set for the option.
 * @param boolean $autoload Whether to autoload the option. Default is null.
 *
 * @return bool True on success, false on failure.
 *
 */
function replace_option(string $option, mixed$value, ?bool $autoload = null) : bool{
    if( ! get_option( $option ) ) {
        return add_option( $option, $value, null, $autoload );
    }

    return update_option( $option, $value, $autoload );
}

/**
 * @param string $plugin
 * @return boolean
 *
 */
function is_plugin_active(string $plugin) : bool {
    return in_array( $plugin, (array) get_option( 'active_plugins', [] ) );
}

function elementorInstance() : ElementorPlugin
{
    static $e;
    if(!isset($e))
    {
        $e = ElementorPlugin::instance();
    }

    return $e;
}

/**
 * @param int $id
 * @param boolean $withCss
 *
 * @return string|null
 *
 */
function getElementorPageById(int $id, bool $withCss = false) : ?string
{
    return elementorInstance()
        ?->frontend
        ?->get_builder_content($id, $withCss)
        ?? null;
}

/**
 * @param string $content
 * @return object
 *
 */
function getContent(string $content): object
{
    $dom = _domDocument($content);
    $xpath = new DOMXPath($dom);
    $divs = $xpath->query('//div[@data-elementor-type="wp-page"]');
    $divs->item(0)
        ->setAttribute('style', 'min-height: auto;');

    $styleTag = $xpath->query('//style');
    $css = $styleTag->item(0)->textContent;

    $modifiedHtml = $dom->saveHTML($divs->item(0));
    $modifiedHtml = str_replace("\n", '', $modifiedHtml);
    $modifiedHtml = str_replace("\t", '', $modifiedHtml);

    return (object)[
        'content' => $modifiedHtml,
        'css' => $css,
    ];
}

/**
 * @param string $content
 *
 * @return DOMDocument
 *
 */
function _domDocument(string $content) : DOMDocument
{
    $dom = new DOMDocument();
    $dom->loadHTML(
        @mb_convert_encoding(
            $content,
            'HTML-ENTITIES',
            'UTF-8'
        )
    );

    return $dom;
}

/**
 * Renames a WordPress hook.
 *
 * @param string $oldName The current name of the hook.
 * @param string $newName The new name for the hook.
 *
 * @return void
 *
 */
function renameHook(string $oldName, string $newName) : void{
    global $wp_filter;
    if(isset($wp_filter[$oldName])){
        $wp_filter[$newName] = $wp_filter[$oldName];
        unset($wp_filter[$oldName]);
    }
}

/**
 * Converts a size in bytes to a human-readable format.
 *
 * @param int $size Size in bytes.
 * @return string Human-readable size string.
 *
 */
function convert(int $size): string
{
    $unit = array('b','kb','mb','gb','tb','pb');

    if($size == 0){
        return '0 b';
    }

    return @round($size/pow(1024,($i=floor(log($size,1024)))),2).' '.$unit[$i];
}

/**
 * @return string
 *
 */
function generateUuidV4Safe(): string {
    try {
        $data = random_bytes(16);
    } catch (RandomException|Exception $e) {
        // Fallback using less secure method
        $data = '';
        for ($i = 0; $i < 16; $i++) {
            $data .= chr(mt_rand(0, 255));
        }
    }

    // Set version and variant bits
    $data[6] = chr((ord($data[6]) & 0x0f) | 0x40); // version 4
    $data[8] = chr((ord($data[8]) & 0x3f) | 0x80); // variant 10

    return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
}

/**
 * Validates if a string is a valid UUID v4.
 *
 * @param string $uuid The UUID string to validate.
 * @return bool True if valid UUID v4, false otherwise.
 *
 */
function isValidUuidV4(string $uuid): bool {
    return preg_match(
            '/^[0-9a-f]{8}-[0-9a-f]{4}-4[0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i',
            $uuid
        ) === 1;
}


/**
 * Generates a string of HTML attributes from an associative array.
 *
 * Boolean attributes (with true value) are rendered as just the key (e.g., disabled).
 * Other attributes are rendered as key="value", with both key and value HTML-escaped.
 *
 *
 * @param array $attrs Associative array of attributes.
 * @return string HTML attributes string.
 */
function createAttributes(array $attrs): string {
    $html = [];
    foreach ($attrs as $key => $value) {
        if (is_bool($value)) {
            if ($value) {
                $html[] = htmlspecialchars($key, ENT_QUOTES);
            }
        } else {
            $html[] = htmlspecialchars($key, ENT_QUOTES) . '="' . htmlspecialchars($value, ENT_QUOTES) . '"';
        }
    }
    return implode(' ', $html);
}
/** @noinspection DuplicatedCode */
function simpleUpdater(string $url, string $version = '0.0.0', $base = null, ?array $icons = null): object
{
    return new class($url, $version, $base, $icons) {
        private string $url;

        private string $slug;
        private string $version;

        private string $base;

        private ?array $icons = null;

        /** @noinspection DuplicatedCode */
        public function __construct(
            string $url, string
                   $version = '0.0.0',
                   $base = null,
                   $icons = null
        )
        {
            $this->url      = $url;
            $this->base     = $base;
            $this->version  = $version;
            $this->slug     = dirname($this->base);
            $this->icons    = $icons ? (array) $icons : null;

            add_filter('plugins_api', [$this, 'update'], 10, 3);
            add_filter('pre_set_site_transient_update_plugins', [$this, 'pre_set_site_transient_update_plugins']);
        }

        /** @noinspection DuplicatedCode */
        private function fetch() : ?array {
            $response = wp_remote_get($this->url, ['timeout' => 5]);
            if (is_wp_error($response)) {
                return null;
            }

            $meta = json_decode(wp_remote_retrieve_body($response), true);
            if (!$meta){
                return null;
            }

            return $meta;
        }

        /** @noinspection DuplicatedCode */
        public function update(mixed $result, string $action, stdClass $args ) {
            if ($action !== 'plugin_information' || empty($args->slug) || $args->slug !== $this->slug) {
                return $result;
            }

            $meta = $this->fetch();

            if (!$meta) {
                return $result;
            }

            $info                   = new stdClass();
            $info->name             = $meta['name'] ?? 'My Plugin';
            $info->slug             = $this->slug;
            $info->version          = $meta['version'] ?? $this->version;
            $info->author           = $meta['author'] ?? 'CarAds.io';
            $info->homepage         = $meta['homepage'] ?? 'carads.io';
            $info->requires         = $meta['requires'] ?? '';
            $info->tested           = $meta['tested'] ?? '';
            $info->sections         = $meta['sections'] ?? ['description' => $meta['description'] ?? ''];
            $info->download_link    = $meta['download_url'] ?? '';
            $info->timestamp        = isset($meta['timestamp']) ? strtotime($meta['timestamp']) : time();

            return $info;
        }

        /** @noinspection DuplicatedCode */
        public function pre_set_site_transient_update_plugins(mixed $transient) : stdClass
        {
            if (!is_object($transient))
            {
                $transient = new stdClass();
            }

            $meta = $this->fetch();

            if (!$meta || empty($meta['version'])) {
                return $transient;
            }

            $obj = (object)[
                'slug'          => $this->slug,
                'plugin'        => $this->base,
                'new_version'   => $meta['version'],
                'url'           => $meta['homepage'] ?? null,
                'package'       => $meta['download_url'] ?? null,
            ];

            if($this->icons){
                $obj->icons = $this->icons;
            }

            $transient->checked[$this->base] = $meta['version'];

            if (version_compare($this->version, $meta['version'], '<')) {
                if(!isset($transient->response)){
                    $transient->response = [];
                }

                if(isset($data->no_update) && is_array($data->no_update) && array_key_exists($this->base, $data->no_update)){
                    unset($data->no_update[$this->base]);
                }

                $transient->response[$this->base] = $obj;
            } else {
                if (isset($transient->response[$this->base])) {
                    unset($transient->response[$this->base]);
                }

                if(!isset($transient->no_update)){
                    $transient->no_update = [];
                }

                $transient->no_update[$this->base] = $obj;
            }

            return $transient;
        }
    };
}

/**
 * @return string
 */
function simple_cdn_version() : string
{
    static $version;

    if($version){
        return $version;
    }

    $factory = current_factory();

    $script     = isset($factory->settings->script) ? $factory->settings->script : 'master';

    if(($script === 'localhost' || $script === 'local') && !$factory->is_test_ip){
        return ($version = 'master');
    }

    return ($version = $script);
}

/**
 * @return \CarAdsDealership\Factory
 */
function current_factory() : \CarAdsDealership\Factory{
    static $factory;
    if(isset($factory)){
        return $factory;
    }

    $ns = current_namespace();
    return $factory = call_user_func("\\{$ns}\Factory::getInstance");
}

/**
 * @return mixed|string
 *
 */
function current_namespace(): mixed
{
    static $ns;
    if(!isset($ns)){
        // get current namespace by __NAMESPACE__ and remove Shared\
        $ns = explode("\\", __NAMESPACE__)[0];
    }

    return $ns;
}

/**
 * @param string $filename
 * @return string
 *
 */
function css_inject_helper(string $filename) : string{
    static $time = null;
    $version = simple_cdn_version();

    if($time === null){
        $time = time();
    }

    return (CA_TEST || ($version === 'localhost' || $version === 'local')) ?
        ("http://localhost:3000/src/style/{$filename}?v={$time}&wx=" . CA_NEXTGEN_VERSION) :
        ("https://js.nextgen.carads.io/{$version}/assets/{$filename}?v={$time}&wx=" . CA_NEXTGEN_VERSION);
}

/**
 * @return void
 *
 */
function elementor_reload_preview() : void{
    $inlineScript = <<<SCRIPT
        (function($){
            $(window).on('elementor/frontend/init', function () {
                setTimeout(function(){
                    window.__carads_nextgen_init();
                }, 1000);
                elementorFrontend.hooks.addAction( 'frontend/element_ready/global', function( x ) {
                    window.__carads_nextgen_init();
                } );
            });
        })(jQuery);
SCRIPT;

    wp_register_script('carads-inline-editor',false,[ 'jquery', 'elementor-frontend' ],null,true);
    wp_enqueue_script('carads-inline-editor');
    wp_add_inline_script('carads-inline-editor', $inlineScript, 'after');
}

/**
 * @param string $url
 * @return string
 *
 */
function fetch_pre_rendered(string $url): string
{
    try {
        $data = (new \WP_Http())->get($url);

        $body = $data['body'] ?? '';

        if ($data['response']['code'] !== 200) {
            return '';
        }

        $dom = _domDocument($body);
        $xpath = new DOMXPath($dom);
        $elements = $xpath->query('//*[@id="__nuxt"]/*');
        return $elements ? implode('', array_map(fn($el) => $el->ownerDocument->saveHTML($el), iterator_to_array($elements))) : '';
    } catch (Exception $e) {
        return '';
    }
}