<?php
/**
 * @copyright  Copyright (C) 2026 CarAds. All rights reserved.
 * @version 2026.01.20082522
 * @author CarAds Team
 */
namespace CarAdsDealership\Controller;
use CarAdsDealership\Basic\BasicController;
use CarAdsNextgen\Utils\Cache;
use stdClass;
use function CarAdsDealership\Basic\css_inject_helper;
use function CarAdsDealership\Basic\render;
use function CarAdsDealership\factory;

class DealerController extends Basic
{
    function render() : void
    {
        $settings   = factory()->configs->get('dealer');
        $slug       = $this->route->active()->params?->slug ?? null;

        // support for debug mode with ?debug=1 - so we can see what is wrong
        $debug      = is_user_logged_in() || isset($_GET['debug']);
        $cleanCache = isset($_GET['clean_cache']) ? 0 : (30*5);

        $dealer = Cache::byReturn(function() use($settings, $slug){
            return $this->fetch($settings->cid, $slug);
        }, !$debug ? ('dealer_' . $slug . '.json') : null, $cleanCache, Cache::ENCODE_JSON);

        if (($dealer?->statusCode??200) === 405){
            $this->seo->robots = 'noindex, nofollow';
            $this->query->set_404();
            status_header(404);
            return;
        }

        $this->query->is_404 = false;
        status_header(200, 'OK');
        set_query_var('ca_dealer', $dealer);

        $this->seo->robots = ($settings->robots_dealer ?? 'noindex, nofollow');
        $this->title($dealer);
        $this->image($dealer);
        $this->description($dealer);
        $this->canonical();

        wp_enqueue_style('car-ads-department', css_inject_helper('department/department.css'), []);
        wp_enqueue_style('car-ads-opening-times', css_inject_helper('department/opening_times/basic.css'), []);
        wp_enqueue_style('car-ads-department-map', css_inject_helper('department/maps.css'), []);

        get_header();

        $list_settings = $settings->app ?? new stdClass();

        if($settings?->app ?? null){
            unset($settings->app); // so we don't send it to javascript again
        }

        if(factory()->configs->exists("dealer_search_{$dealer->did}")){
            $list_settings = json_decode(factory()->configs->getRaw("dealer_search_{$dealer->did}"));
        }
        else{
            $list_settings->department = $dealer->did;
        }

        $url = rtrim(($_SERVER['REDIRECT_URL'] ?? $_SERVER['REQUEST_URI']), '/');

        $list_settings->base = $url;
        $list_settings->pageKey = '_page';
        $list_settings->debug = false;

        if($debug){
            $list_settings->debug = true;
        }

        echo $this->innerHtml(
            slug: $slug,
            settings: $settings,
            dealer: $dealer,
            list_settings: $list_settings
        );

        get_footer();
        exit;
    }

    private function innerHtml(string $slug, stdClass $settings, stdClass $dealer, stdClass $list_settings) : string{
        $settingsString         = json_encode($settings);
        $dealerString           = json_encode($dealer);
        $listSettingsString     = json_encode($list_settings);
        $formSetting            = json_encode([
            'cid' => $settings->cid,
            'did' => $dealer->did,
            'slug' => 'contactdealer',
            'layout' => 'floating-outlined'
        ]);

        return <<<HTML
            <div class="carads">
                <div class="carads-dealer-wrapper">
                    <script type="application/json" id="dealer-settings">
                        {$settingsString}
                    </script>
                    <script type="application/json" id="dealer">
                        {$dealerString}
                    </script>
                    <script type="application/json" id="carads_search_settings">
                        {$listSettingsString}
                    </script>
                    <div id="dealer-app" data-version="2"></div>
                    <div id="app-slow"></div>
                    <div class="ca-dealer-form-wrap carads" id="dealer-form">
                        <div class="ca-dealer-form-wrap-inner max-container ca-flex ca-flex-col ca-gap-4">
                            <h4 class="ca-dealer-form-h4 ca-m-0 ca-text-2xl ca-font-bold ca-pb-4">
                                Skriv til forhandler
                            </h4>
                            <div class="carads-f ca-container ca-dealer-form" data-settings='{$formSetting}'></div>
                        </div>
                    </div>
                </div>
            </div>
HTML;
    }

    private function fetch($id, $slug){
        $url = "https://nextgen.carads.io/pages/{$id}/{$slug}/environment";

        $data = (new \WP_Http())->get($url);

        return json_decode($data['body'] ?? ((object) []));
    }

    private function canonical(): void
    {
        $uri = parse_url(get_site_url() . $_SERVER['REQUEST_URI']);
        $url = (rtrim("{$uri['scheme']}://{$uri['host']}{$uri['path']}", '/'));

        $this->seo->canonical = strtolower($url);
    }

    private function image(stdClass $dealer): void
    {
        $logo = $dealer?->lightlogo ?? null;

        $logo && $this->seo->images = ['https://nextgen.carads.io/media/' . $logo];
    }

    private function title(stdClass $dealer): void
    {
        $title = $dealer?->meta?->title ?? $dealer?->name ?? 'Forhandler';

        $this->seo->title = $title;
    }

    private function description(stdClass $dealer): void
    {
        $description = $dealer?->meta?->description ?? $dealer?->name ?? null;

        if($description){
            $this->seo->description = $description;
        }
    }
}