<?php
/**
 * @copyright  Copyright (C) 2026 CarAds. All rights reserved.
 * @version 2026.01.20082522
 * @author CarAds Team
 */
namespace CarAdsDealership\Controller;

use CarAdsNextgen\Utils\Cache;
use function CarAdsDealership\Basic\css_inject_helper;
use function CarAdsDealership\factory;

class DealersController extends Basic
{
    public function render() : void
    {
        $settings   = factory()->configs->get('dealer');

        // support for debug mode with ?debug=1 - so we can see what is wrong
        $debug      = is_user_logged_in() || isset($_GET['debug']);
        $cleanCache = isset($_GET['clean_cache']) ? 0 : (30*5);
        $dealers = Cache::byReturn(function() use($settings){
            return $this->fetch($settings->cid);
        }, !$debug ? ('dealers.json') : null, $cleanCache, Cache::ENCODE_JSON);

        status_header(200, 'OK');

        $this->seo->title   = $settings?->title ?? 'Forhandlere';
        $this->seo->robots  = $settings->robots ?? 'noindex, nofollow';
        $this->description($dealers);

        $this->query->is_404 = false;

        wp_enqueue_style('car-ads-department', css_inject_helper('department/department.css'), []);
        wp_enqueue_style('car-ads-opening-times', css_inject_helper('department/opening_times/basic.css'), []);
        wp_enqueue_style('car-ads-department-map', css_inject_helper('department/maps.css'), []);

        get_header();

        if(isset($settings->app)){
            unset($settings->app); // so we don't send it to javascript again
        }

        $dealers = array_map(function($dealer){
            return (object)[
                'cached' => true,
                'lightlogo' => ($dealer?->lightlogo ?? null),
                'name' => $dealer->name,
                'meta' => $dealer->meta,
            ];
        }, $dealers);

        echo $this->innerHtml($dealers, $settings);

        get_footer();
        exit;
    }

    private function fetch($id)
    {
        $url = "https://nextgen.carads.io/pages/{$id}/environment";

        $data = (new \WP_Http())->get($url);
        $data = json_decode($data['body'] ?? ((object) []));

        return $data->departments ?? [];
    }

    private function description(array $dealers) : void{
        $descriptions = [];

        foreach ($dealers as $dealer)
        {
            $descriptions[] = $dealer->name;
        }

        $description = implode(', ', $descriptions);

        $this->seo->description = $description;
    }

    private function innerHtml(object|array $dealers, object|array $settings) : string{
        $settings = json_encode($settings);
        $dealers = json_encode($dealers);
        return <<<HTML
        <div class="carads">
            <script type="application/json" id="carads_dealers">
                {$dealers}
            </script>
            <script type="application/json" id="carads_dealers_settings">
                {$settings}
            </script>
            <div id="carads-dealers" data-version="2" class="ca-relative ca-min-h-screen" style="min-height: 100rem">
                &nbsp;
            </div>
        </div>
HTML;
    }
}