<?php
namespace CarAdsDealership\Utils;
use CarAdsDealership\Basic\BasicSettings;
use Exception;
use function CarAdsDealership\factory;
use function CarAdsNextgen\factory as ca_factory;
use function CarAdsLite\factory as lite_factory;

/**
 * Dealership is only work if carads or carads lite is active plugin on website that it use carads for add javascript api.
 *
 * @property-read int $cid
 * @property-read string $api_key
 * @property-read string $secure_key
 */
class Settings extends BasicSettings
{
    protected array $settings = [
        'secure_key' => ['CarAdsDealershipSecureKey', '', 'string', 1],
        'api_key'   => ['CarAdsDealershipApiKey', '', 'string', 1],
        'cid'       => ['CarAdsDealershipClientID', -1, 'int', 1],
    ];
    /**
     * Magic getter that delegates certain settings to the active CarAds implementation.
     *
     * Behavior:
     * - `api_key` and `secure_key`
     *     * If the full CarAds plugin is active (`factory()->carads_exists`) delegate to
     *       the CarAds nextgen settings (`ca_factory()->settings`).
     *     * Otherwise fall back to this instance's settings (`parent::__get`).
     *
     * - `cid`
     *     * Prefer the full CarAds plugin settings when present.
     *     * If full CarAds is not present but CarAds Lite is active (`factory()->lite_exists`),
     *       delegate to the lite settings (`lite_factory()->settings`).
     *     * If neither provider is available, return `null` because no client id can be resolved.
     *
     * Rationale:
     * - The Dealership plugin must use the same API credentials and client id as the primary
     *   CarAds implementation to ensure consistent JavaScript API integration and to avoid
     *   duplicated or conflicting configuration across plugins.
     * - Delegation provides a single source of truth while allowing this plugin to keep its
     *   own fallback values when CarAds is not available.
     *
     * @param string $name Requested property name.
     * @return mixed|null Requested value, or `null` for `cid` if no provider exists.
     * @throws Exception Propagated from parent or delegated getters.
     */
    public function __get($name){
        switch($name){
            case 'api_key':
            case 'secure_key':
                if(factory()->carads_exists){
                    return ca_factory()->settings->__get($name);
                }
                return parent::__get($name);
            case 'cid':
                if(factory()->carads_exists){
                    return ca_factory()->settings->__get($name);
                } else if(factory()->lite_exists) {
                    return lite_factory()->settings->__get($name);
                }
                return null;
            default:
                return parent::__get($name);
        }
    }

    public function update(string $name, mixed $value, $autoload = null, $force = true): void
    {
        switch($name){
            case 'api_key':
            case 'secure_key':
                if(factory()->carads_exists){
                    ca_factory()->settings->update($name, $value, $autoload, $force);
                    return;
                }
                parent::update($name, $value, $autoload, $force);
                return;
            case 'cid':
                if(factory()->carads_exists){
                    ca_factory()->settings->update($name, $value, $autoload, $force);
                    return;
                } else if(factory()->lite_exists) {
                    lite_factory()->settings->update($name, $value, $autoload, $force);
                    return;
                }
                parent::update($name, $value, $autoload, $force);
                return;
            default:
                parent::update($name, $value, $autoload, $force);
                return;
        }
    }
}